"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAlertsForSpace = void 0;
var _lodash = require("lodash");
var _alert_deletion_client = require("../alert_deletion_client");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteAlertsForSpace = async (context, settings, spaceId, abortController) => {
  const taskManager = await context.taskManagerStartPromise;
  const {
    isActiveAlertDeleteEnabled,
    isInactiveAlertDeleteEnabled,
    activeAlertDeleteThreshold,
    inactiveAlertDeleteThreshold,
    categoryIds
  } = settings;
  if (categoryIds && categoryIds.length > 0) {
    if (categoryIds.some(category => !_alert_deletion_client.allowedAppCategories.includes(category))) {
      return {
        numAlertsDeleted: 0,
        errors: [`Invalid category ID found - ${categoryIds} - not deleting alerts`]
      };
    }
  }
  const ruleTypes = context.ruleTypeRegistry.getFilteredTypes({
    categories: categoryIds,
    excludeInternallyManaged: true
  });
  const indices = context.getAlertIndicesAlias(ruleTypes, spaceId);
  let numAlertsDeleted = 0;
  const errors = [];
  if (indices.length === 0) {
    context.logger.warn(`No indices found for rules settings ${settings}. No alerts deleted`);
    return {
      numAlertsDeleted,
      errors: [`No indices found`]
    };
  }
  if (isActiveAlertDeleteEnabled) {
    const activeAlertsQuery = (0, _.getActiveAlertsQuery)(activeAlertDeleteThreshold, spaceId);
    try {
      const {
        numAlertsDeleted: numActiveAlertsDeleted,
        taskIds,
        alertUuidsToClear,
        errors: activeAlertDeletionErrors
      } = await (0, _.deleteAlertsForQuery)(context, indices, activeAlertsQuery, abortController);
      numAlertsDeleted += numActiveAlertsDeleted;
      errors.push(...activeAlertDeletionErrors);
      await taskManager.bulkUpdateState(Array.from(taskIds), state => {
        try {
          const updatedAlertInstances = (0, _lodash.omitBy)(state.alertInstances, ({
            meta: {
              uuid
            }
          }) => alertUuidsToClear.includes(uuid));
          return {
            ...state,
            alertInstances: updatedAlertInstances
          };
        } catch (err) {
          return state;
        }
      });
    } catch (err) {
      const errMessage = `Error deleting active alerts: ${err.message}`;
      context.logger.error(errMessage, {
        error: {
          stack_trace: err.stack
        }
      });
      errors.push(errMessage);
    }
  }
  if (isInactiveAlertDeleteEnabled) {
    const inactiveAlertsQuery = (0, _.getInactiveAlertsQuery)(inactiveAlertDeleteThreshold, spaceId);
    try {
      const {
        numAlertsDeleted: numInactiveAlertsDeleted,
        errors: inactiveAlertDeletionErrors
      } = await (0, _.deleteAlertsForQuery)(context, indices, inactiveAlertsQuery, abortController);
      numAlertsDeleted += numInactiveAlertsDeleted;
      errors.push(...inactiveAlertDeletionErrors);
    } catch (err) {
      const errMessage = `Error deleting inactive alerts: ${err.message}`;
      context.logger.error(errMessage, {
        error: {
          stack_trace: err.stack
        }
      });
      errors.push(errMessage);
    }
  }
  return {
    numAlertsDeleted,
    ...(errors.length > 0 ? {
      errors
    } : {})
  };
};
exports.deleteAlertsForSpace = deleteAlertsForSpace;