"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _security = require("../../services/security");
var _api_privileges = require("../../constants/api_privileges");
var _constants = require("../../../common/constants");
var _constants2 = require("../../constants");
var _types = require("../../types");
var _security2 = require("../../services/security/security");
var _errors = require("../schema/errors");
var _utils = require("../schema/utils");
var _handlers = require("./handlers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  // List
  router.versioned
  // @ts-ignore https://github.com/elastic/kibana/issues/203170
  .get({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.LIST_PATTERN,
    // TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
    fleetAuthz: fleetAuthz => (0, _security2.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('get', _constants2.PACKAGE_POLICY_API_ROUTES.LIST_PATTERN)).granted,
    summary: 'Get package policies',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetPackagePoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => (0, _utils.ListResponseSchema)(_types.PackagePolicyResponseSchema)
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getPackagePoliciesHandler);

  // Get bulk
  router.versioned
  // @ts-ignore https://github.com/elastic/kibana/issues/203170
  .post({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.BULK_GET_PATTERN,
    // TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
    fleetAuthz: fleetAuthz => (0, _security2.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('post', _constants2.PACKAGE_POLICY_API_ROUTES.BULK_GET_PATTERN)).granted,
    summary: 'Bulk get package policies',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.BulkGetPackagePoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.BulkGetPackagePoliciesResponseBodySchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        404: {
          description: 'Not found.',
          body: _errors.notFoundResponse
        }
      }
    }
  }, _handlers.bulkGetPackagePoliciesHandler);

  // Get one
  router.versioned
  // @ts-ignore https://github.com/elastic/kibana/issues/203170
  .get({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.INFO_PATTERN,
    // TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
    fleetAuthz: fleetAuthz => (0, _security2.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('get', _constants2.PACKAGE_POLICY_API_ROUTES.INFO_PATTERN)).granted,
    summary: 'Get a package policy',
    description: 'Get a package policy by ID.',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOnePackagePolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.object({
            item: _types.PackagePolicyResponseSchema
          })
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        404: {
          description: 'Not found.',
          body: _errors.notFoundResponse
        }
      }
    }
  }, _handlers.getOnePackagePolicyHandler);
  router.versioned
  // @ts-ignore https://github.com/elastic/kibana/issues/203170
  .get({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.ORPHANED_INTEGRATION_POLICIES,
    fleetAuthz: {
      integrations: {
        readIntegrationPolicies: true
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {},
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.OrphanedPackagePoliciesResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getOrphanedPackagePolicies);

  // Create
  router.versioned
  // @ts-ignore Authz check moved to service here: https://github.com/elastic/kibana/pull/140458
  .post({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.CREATE_PATTERN,
    summary: 'Create a package policy',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CreatePackagePolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.CreatePackagePolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        409: {
          description: 'A conflict occurred.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.createPackagePolicyHandler);

  // Update
  router.versioned
  // @ts-ignore https://github.com/elastic/kibana/issues/203170
  .put({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.UPDATE_PATTERN,
    // TODO move to kibana authz https://github.com/elastic/kibana/issues/203170
    fleetAuthz: fleetAuthz => (0, _security2.calculateRouteAuthz)(fleetAuthz, (0, _security.getRouteRequiredAuthz)('put', _constants2.PACKAGE_POLICY_API_ROUTES.UPDATE_PATTERN)).granted,
    summary: 'Update a package policy',
    description: 'Update a package policy by ID.',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.UpdatePackagePolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.object({
            item: _types.PackagePolicyResponseSchema
          })
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        403: {
          description: 'Forbidden.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.updatePackagePolicyHandler);

  // Delete (bulk)
  router.versioned.post({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
      }
    },
    summary: 'Bulk delete package policies',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeletePackagePoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DeletePackagePoliciesResponseBodySchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deletePackagePolicyHandler);
  router.versioned.delete({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
      }
    },
    summary: 'Delete a package policy',
    description: 'Delete a package policy by ID.',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteOnePackagePolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DeleteOnePackagePolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deleteOnePackagePolicyHandler);

  // Upgrade
  router.versioned.post({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.UPGRADE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.ALL]
      }
    },
    summary: 'Upgrade a package policy',
    description: 'Upgrade a package policy to a newer package version.',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.UpgradePackagePoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.UpgradePackagePoliciesResponseBodySchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.upgradePackagePolicyHandler);

  // Upgrade - DryRun
  router.versioned.post({
    path: _constants2.PACKAGE_POLICY_API_ROUTES.DRYRUN_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.INTEGRATIONS.READ]
      }
    },
    summary: 'Dry run a package policy upgrade',
    options: {
      tags: ['oas-tag:Fleet package policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DryRunPackagePoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DryRunPackagePoliciesResponseBodySchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.dryRunUpgradePackagePolicyHandler);
};
exports.registerRoutes = registerRoutes;