"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enrichMetricFields = enrichMetricFields;
exports.generateMapKey = generateMapKey;
exports.sampleMetricMetadata = sampleMetricMetadata;
var _otelSemanticConventions = require("@kbn/otel-semantic-conventions");
var _esQuery = require("@kbn/es-query");
var _extract_dimensions = require("../dimensions/extract_dimensions");
var _normalize_unit = require("./normalize_unit");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isErrorResponseBase(subject) {
  return typeof subject === 'object' && subject !== null && 'error' in subject;
}
function generateMapKey(indexName, fieldName) {
  return `${fieldName}>${indexName}`;
}
function buildMetricMetadataMap(response, metricFields, logger) {
  const entries = new Map(metricFields.map(({
    name,
    index: indexName
  }, index) => {
    var _searchResult$hits$hi, _searchResult$hits$hi2;
    const searchResult = response.responses[index];
    const mapKey = generateMapKey(indexName, name);
    if (isErrorResponseBase(searchResult)) {
      logger.error(`Error sampling document for metric ${name}: ${searchResult.error}`);
      return [mapKey, {
        dimensions: []
      }];
    }
    if (!(searchResult !== null && searchResult !== void 0 && (_searchResult$hits$hi = searchResult.hits.hits) !== null && _searchResult$hits$hi !== void 0 && _searchResult$hits$hi.length)) {
      return [mapKey, {
        dimensions: []
      }];
    }
    const fields = (_searchResult$hits$hi2 = searchResult.hits.hits[0].fields) !== null && _searchResult$hits$hi2 !== void 0 ? _searchResult$hits$hi2 : {};
    const {
      dimensions,
      unitFromSample
    } = Object.entries(fields).reduce((acc, [fieldName, fieldValue]) => {
      if (fieldName === _otelSemanticConventions.semconvFlat.unit.name) {
        const value = Array.isArray(fieldValue) ? fieldValue[0] : fieldValue;
        if (typeof value === 'string') {
          acc.unitFromSample = value;
        }
      } else {
        acc.dimensions.push(fieldName);
      }
      return acc;
    }, {
      dimensions: [],
      unitFromSample: undefined
    });
    return [mapKey, {
      dimensions,
      unitFromSample
    }];
  }));
  return new Map(entries);
}
async function sampleMetricMetadata({
  esClient,
  metricFields,
  logger,
  timerange: {
    from,
    to
  }
}) {
  if (metricFields.length === 0) {
    return new Map();
  }
  try {
    const body = [];
    for (const {
      name: field,
      index,
      dimensions
    } of metricFields) {
      body.push({
        index
      });
      // Body for each search
      body.push({
        size: 1,
        terminate_after: 1,
        query: {
          bool: {
            filter: [{
              exists: {
                field
              }
            }, ...(0, _esQuery.dateRangeQuery)(from, to)]
          }
        },
        _source: false,
        fields: dimensions.map(dimension => dimension.name).concat(_otelSemanticConventions.semconvFlat.unit.name)
      });
    }
    const response = await esClient.msearch('sample_metrics_documents', {
      body
    });
    return buildMetricMetadataMap(response, metricFields, logger);
  } catch (error) {
    const metricMetadataMap = new Map();
    for (const {
      name
    } of metricFields) {
      metricMetadataMap.set(name, {
        dimensions: []
      });
    }
    return metricMetadataMap;
  }
}
async function enrichMetricFields({
  esClient,
  metricFields,
  dataStreamFieldCapsMap,
  logger,
  timerange
}) {
  if (metricFields.length === 0) {
    return metricFields;
  }
  const metricMetadataMap = await sampleMetricMetadata({
    esClient,
    metricFields,
    logger,
    timerange
  });
  const uniqueDimensionSets = new Map();
  return metricFields.map(field => {
    var _field$unit;
    const {
      dimensions,
      unitFromSample
    } = metricMetadataMap.get(generateMapKey(field.index, field.name)) || {};
    const fieldCaps = dataStreamFieldCapsMap.get(field.index);
    if (!dimensions || dimensions.length === 0) {
      return {
        ...field,
        dimensions: [],
        noData: true
      };
    }
    const cacheKey = [...dimensions].sort().join(',');
    if (!uniqueDimensionSets.has(cacheKey) && fieldCaps) {
      uniqueDimensionSets.set(cacheKey, (0, _extract_dimensions.extractDimensions)(fieldCaps, dimensions));
    }
    return {
      ...field,
      dimensions: uniqueDimensionSets.get(cacheKey),
      noData: false,
      unit: (0, _normalize_unit.normalizeUnit)({
        fieldName: field.name,
        unit: (_field$unit = field.unit) !== null && _field$unit !== void 0 ? _field$unit : unitFromSample
      })
    };
  });
}