"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeAllFieldsWithSource = void 0;
var _robust_field_access = require("../utils/robust_field_access");
var _filter_field_entry = require("../utils/filter_field_entry");
var _is_objectlike_or_array_of_objectlikes = require("../utils/is_objectlike_or_array_of_objectlikes");
var _is_nested_object = require("../utils/is_nested_object");
var _recursive_unboxing_fields = require("../utils/recursive_unboxing_fields");
var _is_primitive = require("../utils/is_primitive");
var _is_array_of_primitives = require("../utils/is_array_of_primitives");
var _is_type_object = require("../utils/is_type_object");
var _is_path_valid = require("../utils/is_path_valid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Merges all of "doc._source" with its "doc.fields" on a "best effort" basis. See ../README.md for more information
 * on this function and the general strategies.
 *
 * @param doc The document with "_source" and "fields"
 * @param ignoreFields Any fields that we should ignore and never merge from "fields". If the value exists
 * within doc._source it will be untouched and used. If the value does not exist within the doc._source,
 * it will not be added from fields.
 * @returns The two merged together in one object where we can
 */
const mergeAllFieldsWithSource = ({
  doc,
  ignoreFields,
  ignoreFieldsRegexes
}) => {
  var _doc$_source, _doc$fields;
  const source = (_doc$_source = doc._source) !== null && _doc$_source !== void 0 ? _doc$_source : {};
  const fields = (_doc$fields = doc.fields) !== null && _doc$fields !== void 0 ? _doc$fields : {};
  const fieldsKeys = Object.keys(fields);
  fieldsKeys.forEach(fieldsKey => {
    const valueInMergedDocument = (0, _robust_field_access.robustGet)({
      key: fieldsKey,
      document: source
    });
    const fieldsValue = fields[fieldsKey];
    if (!hasEarlyReturnConditions({
      fieldsValue,
      fieldsKey,
      merged: source
    }) && (0, _filter_field_entry.filterFieldEntry)([fieldsKey, fieldsValue], fieldsKeys, ignoreFields, ignoreFieldsRegexes)) {
      if (valueInMergedDocument === undefined) {
        const valueToMerge = (0, _recursive_unboxing_fields.recursiveUnboxingFields)(fieldsValue, valueInMergedDocument);
        (0, _robust_field_access.robustSet)({
          key: fieldsKey,
          valueToSet: valueToMerge,
          document: source
        });
      } else if ((0, _is_primitive.isPrimitive)(valueInMergedDocument)) {
        const valueToMerge = (0, _recursive_unboxing_fields.recursiveUnboxingFields)(fieldsValue, valueInMergedDocument);
        (0, _robust_field_access.robustSet)({
          key: fieldsKey,
          valueToSet: valueToMerge,
          document: source
        });
      } else if ((0, _is_array_of_primitives.isArrayOfPrimitives)(valueInMergedDocument)) {
        const valueToMerge = (0, _recursive_unboxing_fields.recursiveUnboxingFields)(fieldsValue, valueInMergedDocument);
        (0, _robust_field_access.robustSet)({
          key: fieldsKey,
          valueToSet: valueToMerge,
          document: source
        });
      } else if ((0, _is_objectlike_or_array_of_objectlikes.isObjectLikeOrArrayOfObjectLikes)(valueInMergedDocument) && (0, _is_nested_object.isNestedObject)(fieldsValue) && !Array.isArray(valueInMergedDocument)) {
        const valueToMerge = (0, _recursive_unboxing_fields.recursiveUnboxingFields)(fieldsValue, valueInMergedDocument);
        (0, _robust_field_access.robustSet)({
          key: fieldsKey,
          valueToSet: valueToMerge,
          document: source
        });
      } else if ((0, _is_objectlike_or_array_of_objectlikes.isObjectLikeOrArrayOfObjectLikes)(valueInMergedDocument) && (0, _is_nested_object.isNestedObject)(fieldsValue) && Array.isArray(valueInMergedDocument)) {
        const valueToMerge = (0, _recursive_unboxing_fields.recursiveUnboxingFields)(fieldsValue, valueInMergedDocument);
        (0, _robust_field_access.robustSet)({
          key: fieldsKey,
          valueToSet: valueToMerge,
          document: source
        });
      }
    }
  });
  return {
    ...doc,
    _source: source
  };
};

/**
 * Returns true if any early return conditions are met which are
 *   - If the fieldsValue is an empty array return
 *   - If we have an array within the path return and the value in our merged documented is non-existent
 *   - If the value is an object or is an array of object types and we don't have a nested field
 * @param fieldsValue The field value to check
 * @param fieldsKey The key of the field we are checking
 * @param merged The merge document which is what we are testing conditions against
 * @returns true if we should return early, otherwise false
 */
exports.mergeAllFieldsWithSource = mergeAllFieldsWithSource;
const hasEarlyReturnConditions = ({
  fieldsValue,
  fieldsKey,
  merged
}) => {
  const valueInMergedDocument = (0, _robust_field_access.robustGet)({
    key: fieldsKey,
    document: merged
  });
  return fieldsValue.length === 0 || valueInMergedDocument === undefined && !(0, _is_path_valid.robustIsPathValid)(fieldsKey, merged) || (0, _is_objectlike_or_array_of_objectlikes.isObjectLikeOrArrayOfObjectLikes)(valueInMergedDocument) && !(0, _is_nested_object.isNestedObject)(fieldsValue) && !(0, _is_type_object.isTypeObject)(fieldsValue);
};