"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.uploadUsersCSVRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../../../../common/entity_analytics/privileged_user_monitoring/constants");
var _constants2 = require("../../../../../../common/constants");
var _assert_advanced_setting_enabled = require("../../../utils/assert_advanced_setting_enabled");
var _csv_upload = require("../../users/csv_upload");
var _check_and_init_privmon_resources = require("../../check_and_init_privmon_resources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const uploadUsersCSVRoute = (router, logger, config) => {
  router.versioned.post({
    access: 'public',
    path: _constants.MONITORING_USERS_CSV_UPLOAD_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants2.APP_ID}-entity-analytics`]
      }
    },
    options: {
      body: {
        output: 'stream',
        accepts: 'multipart/form-data',
        maxBytes: _constants.PRIVMON_USERS_CSV_MAX_SIZE_BYTES_WITH_TOLERANCE
      }
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: _configSchema.schema.object({
          file: _configSchema.schema.stream()
        })
      }
    }
  }, async (context, request, response) => {
    const {
      errorRetries,
      maxBulkRequestBodySizeBytes
    } = config.entityAnalytics.monitoring.privileges.users.csvUpload;
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      await (0, _assert_advanced_setting_enabled.assertAdvancedSettingsEnabled)(await context.core, _constants2.ENABLE_PRIVILEGED_USER_MONITORING_SETTING);
      const secSol = await context.securitySolution;
      const fileStream = request.body.file;
      const dataClient = secSol.getPrivilegeMonitoringDataClient();
      const csvService = (0, _csv_upload.createPrivilegedUsersCsvService)(dataClient);
      await (0, _check_and_init_privmon_resources.checkAndInitPrivilegeMonitoringResources)(context, logger);
      const body = await csvService.bulkUpload(fileStream, {
        retries: errorRetries,
        flushBytes: maxBulkRequestBodySizeBytes
      });
      return response.ok({
        body
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error uploading users via CSV: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.uploadUsersCSVRoute = uploadUsersCSVRoute;