"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BulkPurgeRollupData = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _eui = require("@elastic/eui");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class BulkPurgeRollupData {
  constructor(esClient, repository) {
    this.esClient = esClient;
    this.repository = repository;
  }
  async execute(params) {
    const lookback = this.getTimestamp(params.purgePolicy);
    const slos = await this.repository.findAllByIds(params.list);
    if (params.force !== true) {
      await this.validatePurgePolicy(slos, params.purgePolicy);
    }
    const response = await this.esClient.deleteByQuery({
      index: _constants.SLI_DESTINATION_INDEX_PATTERN,
      refresh: false,
      wait_for_completion: false,
      conflicts: 'proceed',
      slices: 'auto',
      query: {
        bool: {
          filter: [{
            terms: {
              'slo.id': slos.map(slo => slo.id)
            }
          }, {
            range: {
              '@timestamp': {
                lte: lookback
              }
            }
          }]
        }
      }
    });
    return {
      taskId: response.task
    };
  }
  async validatePurgePolicy(slos, purgePolicy) {
    const purgeType = purgePolicy.purgeType;
    let inputIsInvalid = false;
    switch (purgeType) {
      case 'fixed_age':
        const duration = purgePolicy.age;
        inputIsInvalid = slos.some(slo => {
          if (_sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow)) {
            return (0, _moment.default)(Date.now()).subtract(duration.asSeconds(), 's').isAfter((0, _moment.default)(Date.now()).startOf(slo.timeWindow.duration.unit));
          } else {
            return duration.isShorterThan(slo.timeWindow.duration);
          }
        });
        break;
      case 'fixed_time':
        const timestampMoment = (0, _moment.default)(purgePolicy.timestamp);
        inputIsInvalid = slos.some(slo => {
          if (_sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow)) {
            return timestampMoment.isAfter((0, _moment.default)(Date.now()).startOf(slo.timeWindow.duration.unit));
          } else {
            return timestampMoment.isAfter((0, _moment.default)(Date.now()).subtract(slo.timeWindow.duration.asSeconds(), 's'));
          }
        });
        break;
      default:
        (0, _eui.assertNever)(purgeType);
    }
    if (inputIsInvalid) {
      throw new _errors.IllegalArgumentError(`The provided purge policy is invalid. At least one SLO has a time window that is longer than the provided purge policy.`);
    }
  }
  getTimestamp(purgePolicy) {
    if (purgePolicy.purgeType === 'fixed_age') {
      return `now-${purgePolicy.age.format()}`;
    } else if (purgePolicy.purgeType === 'fixed_time') {
      return purgePolicy.timestamp.toISOString();
    }
    (0, _eui.assertNever)(purgePolicy);
  }
}
exports.BulkPurgeRollupData = BulkPurgeRollupData;