"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetSLOStatsOverview = exports.ES_PAGESIZE_LIMIT = void 0;
var _esQuery = require("@kbn/es-query");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _moment = _interopRequireDefault(require("moment"));
var _queries = require("../utils/queries");
var _slo_settings = require("./slo_settings");
var _transform_generators = require("./transform_generators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ES_PAGESIZE_LIMIT = exports.ES_PAGESIZE_LIMIT = 1000;

/*
    This service retrieves stats from alert and rule indices to display on the SLO landing page. 
    When filters are applied to SLOs, we want to forward those filters onto the searches performed on alerts and rules so the overview stats actively reflect viewable SLO data.
    To achieve this, we need to retrieve a list of all SLO ids and instanceIds that may appear across all SLO list pages
      to use them as filter conditions on the alert and rule stats that we want to count. 
*/

class GetSLOStatsOverview {
  constructor(soClient, scopedClusterClient, spaceId, logger, rulesClient, racClient) {
    this.soClient = soClient;
    this.scopedClusterClient = scopedClusterClient;
    this.spaceId = spaceId;
    this.logger = logger;
    this.rulesClient = rulesClient;
    this.racClient = racClient;
  }
  getAfterKey(agg) {
    if (agg && typeof agg === 'object' && 'after_key' in agg && agg.after_key) {
      return agg.after_key;
    }
    return undefined;
  }
  processSloQueryBuckets(buckets, instanceId) {
    return buckets.map(bucket => {
      return {
        bucketKey: bucket.key.sloId,
        query: {
          bool: {
            must: [{
              term: {
                'kibana.alert.rule.parameters.sloId': bucket.key.sloId
              }
            }, ...(instanceId ? [{
              term: {
                'kibana.alert.instance.id': bucket.key.sloInstanceId
              }
            }] : [])]
          }
        }
      };
    });
  }

  /*
     If we know there are no SLOs that match the provided filters, we can skip querying for rules and alerts
    */
  async fetchRulesAndAlerts({
    querySLOsForIds,
    sloRuleQueryKeys,
    ruleFilters,
    alertFilters
  }) {
    return await Promise.all(querySLOsForIds && sloRuleQueryKeys.length === 0 ? [{
      total: 0
    }, {
      activeAlertCount: 0,
      recoveredAlertCount: 0
    }] : [this.rulesClient.find({
      options: {
        ruleTypeIds: _ruleDataUtils.SLO_RULE_TYPE_IDS,
        consumers: [_ruleDataUtils.AlertConsumers.SLO, _ruleDataUtils.AlertConsumers.ALERTS, _ruleDataUtils.AlertConsumers.OBSERVABILITY],
        ...(ruleFilters ? {
          filter: ruleFilters
        } : {})
      }
    }), this.racClient.getAlertSummary({
      ruleTypeIds: _ruleDataUtils.SLO_RULE_TYPE_IDS,
      consumers: [_ruleDataUtils.AlertConsumers.SLO, _ruleDataUtils.AlertConsumers.ALERTS, _ruleDataUtils.AlertConsumers.OBSERVABILITY],
      gte: (0, _moment.default)().subtract(24, 'hours').toISOString(),
      lte: (0, _moment.default)().toISOString(),
      ...(alertFilters !== null && alertFilters !== void 0 && alertFilters.length ? {
        filter: alertFilters
      } : {})
    })]);
  }
  async execute(params) {
    var _params$kqlQuery, _params$filters, _params$kqlQuery2, _parsedFilters$filter, _parsedFilters$must_n, _aggs$not_stale$viola, _aggs$not_stale, _aggs$not_stale$degra, _aggs$not_stale2, _aggs$not_stale$healt, _aggs$not_stale3, _aggs$not_stale3$heal, _aggs$not_stale$noDat, _aggs$not_stale4, _aggs$stale$doc_count;
    const settings = await (0, _slo_settings.getSloSettings)(this.soClient);
    const {
      indices
    } = await (0, _slo_settings.getSummaryIndices)(this.scopedClusterClient.asInternalUser, settings);
    const kqlQuery = (_params$kqlQuery = params === null || params === void 0 ? void 0 : params.kqlQuery) !== null && _params$kqlQuery !== void 0 ? _params$kqlQuery : '';
    const filters = (_params$filters = params === null || params === void 0 ? void 0 : params.filters) !== null && _params$filters !== void 0 ? _params$filters : '';
    const parsedFilters = (0, _transform_generators.parseStringFilters)(filters, this.logger);
    const kqlQueriesProvided = !!(params !== null && params !== void 0 && params.kqlQuery) && (params === null || params === void 0 ? void 0 : (_params$kqlQuery2 = params.kqlQuery) === null || _params$kqlQuery2 === void 0 ? void 0 : _params$kqlQuery2.length) > 0;

    /*
      If there are any filters or KQL queries provided, we need to query for SLO ids and instanceIds to use as filter conditions on the alert and rule searches.
    */
    const filtersProvided = !!parsedFilters && Object.keys(parsedFilters).some(key => Array.isArray(parsedFilters[key]) && parsedFilters[key].length > 0);
    const querySLOsForIds = filtersProvided || kqlQueriesProvided;
    const sloRuleQueryKeys = [];
    const instanceIdIncluded = Object.values(params).find(value => typeof value === 'string' && value.includes('slo.instanceId'));
    const alertFilterTerms = [];
    let afterKey;
    try {
      if (querySLOsForIds) {
        do {
          var _parsedFilters$must, _sloIdCompositeQueryR, _sloIdCompositeQueryR2, _sloIdCompositeQueryR3;
          const sloIdCompositeQueryResponse = await this.scopedClusterClient.asCurrentUser.search({
            index: indices,
            size: 0,
            aggs: {
              sloIds: {
                composite: {
                  after: afterKey,
                  size: ES_PAGESIZE_LIMIT,
                  sources: [{
                    sloId: {
                      terms: {
                        field: 'slo.id'
                      }
                    }
                  }, ...(instanceIdIncluded ? [{
                    sloInstanceId: {
                      terms: {
                        field: 'slo.instanceId'
                      }
                    }
                  }] : [])]
                }
              }
            },
            query: {
              bool: {
                ...parsedFilters,
                ...(params.kqlQuery ? {
                  must: [...((_parsedFilters$must = parsedFilters.must) !== null && _parsedFilters$must !== void 0 ? _parsedFilters$must : []), {
                    kql: {
                      query: params.kqlQuery
                    }
                  }]
                } : {})
              }
            }
          });
          afterKey = this.getAfterKey((_sloIdCompositeQueryR = sloIdCompositeQueryResponse.aggregations) === null || _sloIdCompositeQueryR === void 0 ? void 0 : _sloIdCompositeQueryR.sloIds);
          const buckets = (_sloIdCompositeQueryR2 = sloIdCompositeQueryResponse.aggregations) === null || _sloIdCompositeQueryR2 === void 0 ? void 0 : (_sloIdCompositeQueryR3 = _sloIdCompositeQueryR2.sloIds) === null || _sloIdCompositeQueryR3 === void 0 ? void 0 : _sloIdCompositeQueryR3.buckets;
          if (buckets) {
            const processedBuckets = this.processSloQueryBuckets(buckets, instanceIdIncluded);
            for (const {
              bucketKey,
              query
            } of processedBuckets) {
              alertFilterTerms.push(query);
              sloRuleQueryKeys.push(bucketKey);
            }
          }
        } while (afterKey);
      }
    } catch (error) {
      this.logger.error(`Error querying SLOs for IDs: ${error}`);
      throw error;
    }
    const response = await (0, _queries.typedSearch)(this.scopedClusterClient.asCurrentUser, {
      index: indices,
      size: 0,
      query: {
        bool: {
          filter: [{
            term: {
              spaceId: this.spaceId
            }
          }, (0, _transform_generators.getElasticsearchQueryOrThrow)(kqlQuery), ...((_parsedFilters$filter = parsedFilters.filter) !== null && _parsedFilters$filter !== void 0 ? _parsedFilters$filter : [])],
          must_not: [...((_parsedFilters$must_n = parsedFilters.must_not) !== null && _parsedFilters$must_n !== void 0 ? _parsedFilters$must_n : [])]
        }
      },
      aggs: {
        stale: {
          filter: {
            range: {
              summaryUpdatedAt: {
                lt: `now-${settings.staleThresholdInHours}h`
              }
            }
          }
        },
        not_stale: {
          filter: {
            range: {
              summaryUpdatedAt: {
                gte: `now-${settings.staleThresholdInHours}h`
              }
            }
          },
          aggs: {
            violated: {
              filter: {
                term: {
                  status: 'VIOLATED'
                }
              }
            },
            healthy: {
              filter: {
                term: {
                  status: 'HEALTHY'
                }
              }
            },
            degrading: {
              filter: {
                term: {
                  status: 'DEGRADING'
                }
              }
            },
            noData: {
              filter: {
                term: {
                  status: 'NO_DATA'
                }
              }
            }
          }
        }
      }
    });
    const ruleFilters = sloRuleQueryKeys.length > 0 ? _esQuery.nodeBuilder.or(sloRuleQueryKeys.map(sloId => _esQuery.nodeBuilder.is(`alert.attributes.params.sloId`, sloId))) : undefined;
    const alertFilters = alertFilterTerms.length > 0 ? [{
      bool: {
        should: [...alertFilterTerms]
      }
    }] : [];
    const [rules, alerts] = await this.fetchRulesAndAlerts({
      querySLOsForIds,
      sloRuleQueryKeys,
      ruleFilters,
      alertFilters
    });
    const aggs = response.aggregations;
    return {
      violated: (_aggs$not_stale$viola = aggs === null || aggs === void 0 ? void 0 : (_aggs$not_stale = aggs.not_stale) === null || _aggs$not_stale === void 0 ? void 0 : _aggs$not_stale.violated.doc_count) !== null && _aggs$not_stale$viola !== void 0 ? _aggs$not_stale$viola : 0,
      degrading: (_aggs$not_stale$degra = aggs === null || aggs === void 0 ? void 0 : (_aggs$not_stale2 = aggs.not_stale) === null || _aggs$not_stale2 === void 0 ? void 0 : _aggs$not_stale2.degrading.doc_count) !== null && _aggs$not_stale$degra !== void 0 ? _aggs$not_stale$degra : 0,
      healthy: (_aggs$not_stale$healt = aggs === null || aggs === void 0 ? void 0 : (_aggs$not_stale3 = aggs.not_stale) === null || _aggs$not_stale3 === void 0 ? void 0 : (_aggs$not_stale3$heal = _aggs$not_stale3.healthy) === null || _aggs$not_stale3$heal === void 0 ? void 0 : _aggs$not_stale3$heal.doc_count) !== null && _aggs$not_stale$healt !== void 0 ? _aggs$not_stale$healt : 0,
      noData: (_aggs$not_stale$noDat = aggs === null || aggs === void 0 ? void 0 : (_aggs$not_stale4 = aggs.not_stale) === null || _aggs$not_stale4 === void 0 ? void 0 : _aggs$not_stale4.noData.doc_count) !== null && _aggs$not_stale$noDat !== void 0 ? _aggs$not_stale$noDat : 0,
      stale: (_aggs$stale$doc_count = aggs === null || aggs === void 0 ? void 0 : aggs.stale.doc_count) !== null && _aggs$stale$doc_count !== void 0 ? _aggs$stale$doc_count : 0,
      burnRateRules: rules.total,
      burnRateActiveAlerts: alerts.activeAlertCount,
      burnRateRecoveredAlerts: alerts.recoveredAlertCount
    };
  }
}
exports.GetSLOStatsOverview = GetSLOStatsOverview;