"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CleanUpTempSummary = void 0;
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CleanUpTempSummary {
  constructor(esClient, logger, abortController) {
    this.esClient = esClient;
    this.logger = logger;
    this.abortController = abortController;
  }
  async execute() {
    const openCircuitBreaker = await this.shouldOpenCircuitBreaker();
    if (openCircuitBreaker) {
      this.logger.debug('No temporary documents found, skipping.');
      return;
    }
    let searchAfterKey;
    do {
      const {
        buckets,
        nextSearchAfterKey
      } = await this.findDuplicateTemporaryDocuments(searchAfterKey);
      searchAfterKey = nextSearchAfterKey;
      if (buckets.length > 0) {
        await this.deleteDuplicateTemporaryDocuments(buckets);
      }
    } while (searchAfterKey);
  }
  async shouldOpenCircuitBreaker() {
    const results = await this.esClient.count({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      terminate_after: 1
    }, {
      signal: this.abortController.signal
    });
    return results.count === 0;
  }
  async findDuplicateTemporaryDocuments(searchAfterKey) {
    var _results$aggregations, _results$aggregations2, _results$aggregations3;
    this.logger.debug('Searching for duplicate temporary documents');
    const results = await this.esClient.search({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      size: 0,
      aggs: {
        duplicate_ids: {
          composite: {
            size: 10000,
            after: searchAfterKey,
            sources: [{
              spaceId: {
                terms: {
                  field: 'spaceId'
                }
              }
            }, {
              id: {
                terms: {
                  field: 'slo.id'
                }
              }
            }]
          },
          aggs: {
            cardinality_istempdoc: {
              cardinality: {
                field: 'isTempDoc'
              }
            },
            find_duplicates: {
              bucket_selector: {
                buckets_path: {
                  cardinality: 'cardinality_istempdoc'
                },
                script: 'params.cardinality == 2'
              }
            }
          }
        }
      }
    }, {
      signal: this.abortController.signal
    });
    const buckets = ((_results$aggregations = (_results$aggregations2 = results.aggregations) === null || _results$aggregations2 === void 0 ? void 0 : _results$aggregations2.duplicate_ids.buckets) !== null && _results$aggregations !== void 0 ? _results$aggregations : []).map(bucket => bucket.key);
    const nextSearchAfterKey = (_results$aggregations3 = results.aggregations) === null || _results$aggregations3 === void 0 ? void 0 : _results$aggregations3.duplicate_ids.after_key;
    this.logger.debug(`Found ${buckets.length} duplicate temporary documents`);
    return {
      buckets,
      nextSearchAfterKey
    };
  }
  async deleteDuplicateTemporaryDocuments(buckets) {
    this.logger.debug(`Deleting ${buckets.length} duplicate temporary documents`);
    await this.esClient.deleteByQuery({
      index: _constants.SUMMARY_TEMP_INDEX_NAME,
      wait_for_completion: false,
      slices: 'auto',
      conflicts: 'proceed',
      query: {
        bool: {
          should: buckets.map(bucket => {
            return {
              bool: {
                must: [{
                  term: {
                    isTempDoc: true
                  }
                }, {
                  term: {
                    'slo.id': bucket.id
                  }
                }, {
                  term: {
                    spaceId: bucket.spaceId
                  }
                }]
              }
            };
          }),
          minimum_should_match: 1
        }
      }
    }, {
      signal: this.abortController.signal
    });
  }
}
exports.CleanUpTempSummary = CleanUpTempSummary;