"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SpaceValidator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _space_identifier_utils = require("./space_identifier_utils");
var _is_reserved_space = require("../../../common/is_reserved_space");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SpaceValidator {
  constructor(options = {}) {
    (0, _defineProperty2.default)(this, "shouldValidate", void 0);
    this.shouldValidate = options.shouldValidate || false;
  }
  enableValidation() {
    this.shouldValidate = true;
  }
  disableValidation() {
    this.shouldValidate = false;
  }
  validateSpaceName(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (!space.name || !space.name.trim()) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.requiredNameErrorMessage', {
        defaultMessage: 'Enter a name.'
      }));
    }
    if (space.name.length > 1024) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.nameMaxLengthErrorMessage', {
        defaultMessage: 'Name must not exceed 1024 characters.'
      }));
    }
    return valid();
  }
  validateSpaceDescription(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (space.description && space.description.length > 2000) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.describeMaxLengthErrorMessage', {
        defaultMessage: 'Description must not exceed 2000 characters.'
      }));
    }
    return valid();
  }
  validateURLIdentifier(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if ((0, _is_reserved_space.isReservedSpace)(space)) {
      return valid();
    }
    if (!space.id) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.urlIdentifierRequiredErrorMessage', {
        defaultMessage: 'Enter a URL identifier.'
      }));
    }
    if (!(0, _space_identifier_utils.isValidSpaceIdentifier)(space.id)) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.urlIdentifierAllowedCharactersErrorMessage', {
        defaultMessage: 'URL identifier can only contain a-z, 0-9, and the characters "_" and "-".'
      }));
    }
    return valid();
  }
  validateAvatarInitials(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (space.avatarType !== 'image') {
      if (!space.initials) {
        return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.requiredInitialsErrorMessage', {
          defaultMessage: 'Enter initials.'
        }));
      }
      if (space.initials.length > 2) {
        return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.maxLengthInitialsErrorMessage', {
          defaultMessage: 'Enter no more than 2 characters.'
        }));
      }
    }
    return valid();
  }
  validateAvatarColor(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (!space.color) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.requiredColorErrorMessage', {
        defaultMessage: 'Select a background color.'
      }));
    }
    if (!(0, _eui.isValidHex)(space.color)) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.invalidColorErrorMessage', {
        defaultMessage: 'Enter a valid HEX color code.'
      }));
    }
    return valid();
  }
  validateAvatarImage(space) {
    if (!this.shouldValidate) {
      return valid();
    }
    if (space.avatarType === 'image' && !space.imageUrl) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.requiredImageErrorMessage', {
        defaultMessage: 'Upload an image.'
      }));
    }
    return valid();
  }
  validateSolutionView(space, isEditing, allowSolutionVisibility = true) {
    if (!this.shouldValidate || isEditing || !allowSolutionVisibility) {
      return valid();
    }
    if (!space.solution) {
      return invalid(_i18n.i18n.translate('xpack.spaces.management.validateSpace.requiredSolutionViewErrorMessage', {
        defaultMessage: 'Select a solution.'
      }));
    }
    return valid();
  }
  validateEnabledFeatures(space) {
    return valid();
  }
  validateForSave(space, isEditing, allowSolutionVisibility) {
    const {
      isInvalid: isNameInvalid
    } = this.validateSpaceName(space);
    const {
      isInvalid: isDescriptionInvalid
    } = this.validateSpaceDescription(space);
    const {
      isInvalid: isIdentifierInvalid
    } = this.validateURLIdentifier(space);
    const {
      isInvalid: isAvatarInitialsInvalid
    } = this.validateAvatarInitials(space);
    const {
      isInvalid: isAvatarColorInvalid
    } = this.validateAvatarColor(space);
    const {
      isInvalid: isAvatarImageInvalid
    } = this.validateAvatarImage(space);
    const {
      isInvalid: areFeaturesInvalid
    } = this.validateEnabledFeatures(space);
    const {
      isInvalid: isSolutionViewInvalid
    } = this.validateSolutionView(space, isEditing, allowSolutionVisibility);
    if (isNameInvalid || isDescriptionInvalid || isIdentifierInvalid || isAvatarInitialsInvalid || isAvatarColorInvalid || isAvatarImageInvalid || areFeaturesInvalid || isSolutionViewInvalid) {
      return invalid();
    }
    return valid();
  }
}
exports.SpaceValidator = SpaceValidator;
function invalid(error = '') {
  return {
    isInvalid: true,
    error
  };
}
function valid() {
  return {
    isInvalid: false
  };
}