"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatUpdateRequest = void 0;
exports.prepareAdditionalFieldsForCreation = prepareAdditionalFieldsForCreation;
exports.transformFieldMetadataToRecord = transformFieldMetadataToRecord;
exports.validateValues = validateValues;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getValueFromOldField = (fieldMeta, value) => {
  switch (fieldMeta.input_type) {
    case 'textarea':
      return value === null || value === undefined ? null : value;
    case 'multiselect':
    case 'select':
    case 'datetimepicker':
    case 'datepicker':
    case 'boolean':
    case 'number':
    case 'text':
      return value;
    default:
      return null;
  }
};
function validateValues(fieldMeta, givenValues) {
  const valuesMeta = fieldMeta.values;
  if (valuesMeta && Array.isArray(valuesMeta)) {
    if (!givenValues.every(id => valuesMeta.some(option => option.value === id))) {
      throw new Error(`Invalid values provided to ${fieldMeta.name}: ${JSON.stringify(givenValues, null, 2)}. Accepted values: ${valuesMeta.map(v => `${v.value} (for "${v.label}")`)}`);
    }
  }
  return true;
}
function getValueFieldShape(fieldMeta, value, oldValue) {
  switch (fieldMeta.input_type) {
    case 'textarea':
      if (value === null || value === undefined) {
        return {
          textarea: null
        };
      } else if (typeof value === 'object' && 'format' in value && 'content' in value) {
        return {
          textarea: {
            ...value
          }
        };
      } else {
        return {
          textarea: {
            format: (0, _lodash.isObject)(oldValue) && 'format' in oldValue ? oldValue === null || oldValue === void 0 ? void 0 : oldValue.format : 'text',
            content: value
          }
        };
      }
    case 'multiselect':
      if (value === null || value === undefined) {
        return {};
      }
      const ids = Array.isArray(value) ? value.map(item => item) : [value];
      if (validateValues(fieldMeta, ids)) {
        return {
          ids
        };
      }
    case 'select':
      if (value === null || value === undefined) {
        return {};
      }
      if (validateValues(fieldMeta, [value])) {
        return {
          id: value
        };
      }
    case 'datetimepicker':
    case 'datepicker':
      return (0, _lodash.isNumber)(value) ? {
        date: value
      } : {};
    case 'boolean':
      return (0, _lodash.isBoolean)(value) ? {
        boolean: value
      } : {};
    case 'number':
      return (0, _lodash.isNumber)(value) ? {
        object: value
      } : {};
    case 'text':
      return (0, _lodash.isString)(value) ? {
        text: value
      } : {};
    default:
      return {};
  }
}
const formatUpdateRequest = ({
  oldIncident,
  newIncident,
  fields
}) => {
  const {
    additionalFields,
    ...root
  } = newIncident;
  // We can merge the root and the additional fields since they are treated the same
  // in update requests to Resilient
  const updates = {
    ...(additionalFields || {}),
    ...root
  };
  const fieldsRecord = transformFieldMetadataToRecord(fields);
  return {
    changes: Object.keys(updates).map(key => {
      let name = key;
      if (key === 'incidentTypes') {
        name = 'incident_type_ids';
      }
      if (key === 'severityCode') {
        name = 'severity_code';
      }
      const fieldMeta = fieldsRecord[name];
      if (!fieldMeta) {
        // if we don't have metadata about the field, we can't process it
        throw new Error(`No metadata found for field ${name}`);
      }
      let oldValue = oldIncident[name] ? oldIncident[name] : oldIncident[key];
      if (fieldMeta.prefix) {
        oldValue = oldIncident[fieldMeta.prefix] ? oldIncident[fieldMeta.prefix][key] : null;
      }
      const newValue = updates[name] ? updates[name] : updates[key];
      const oldValueShape = getValueFieldShape(fieldMeta, getValueFromOldField(fieldMeta, oldValue));
      const newValueShape = getValueFieldShape(fieldMeta, newValue, oldValue);
      return {
        field: {
          name
        },
        old_value: oldValueShape,
        new_value: newValueShape
      };
    })
  };
};
exports.formatUpdateRequest = formatUpdateRequest;
function transformFieldMetadataToRecord(fields) {
  return fields.reduce((acc, field) => {
    acc[field.name] = {
      name: field.name,
      input_type: field.input_type,
      read_only: field.read_only,
      required: field.required,
      text: field.text,
      prefix: field.prefix,
      values: field.values
    };
    return acc;
  }, {});
}
function prepareAdditionalFieldsForCreation(fields, additionalFields) {
  const {
    properties,
    ...rest
  } = additionalFields;
  const flattenedAdditionalFields = {
    ...(properties !== null && properties !== void 0 ? properties : {}),
    ...rest
  };
  const data = {};
  const fieldsMetaData = transformFieldMetadataToRecord(fields);
  Object.entries(flattenedAdditionalFields).forEach(([key, value]) => {
    const fieldMeta = fieldsMetaData[key];

    // validate `select` and `multiselect` values
    if ((fieldMeta === null || fieldMeta === void 0 ? void 0 : fieldMeta.input_type) === 'select' || (fieldMeta === null || fieldMeta === void 0 ? void 0 : fieldMeta.input_type) === 'multiselect') {
      validateValues(fieldMeta, Array.isArray(value) ? value : [value]);
    }

    // Some fields need to be prefixed
    if (fieldMeta && fieldMeta.prefix) {
      if (!data[fieldMeta.prefix]) {
        data[fieldMeta.prefix] = {};
      }
      data[fieldMeta.prefix][key] = value;
    } else {
      data[key] = value;
    }
  });
  return data;
}