"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAxiosConfig = void 0;
var _axios = _interopRequireDefault(require("axios"));
var _get_oauth_client_credentials_access_token = require("@kbn/actions-plugin/server/lib/get_oauth_client_credentials_access_token");
var _lib = require("@kbn/actions-plugin/server/lib");
var _oauth2_delete_token_axios_interceptor = require("../../../common/auth/oauth2_delete_token_axios_interceptor");
var _utils = require("../../../common/auth/utils");
var _constants = require("../../../common/auth/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getOAuth2AxiosConfig = async ({
  connectorId,
  config,
  secrets,
  services: {
    connectorTokenClient
  },
  logger,
  configurationUtilities
}) => {
  const {
    accessTokenUrl,
    clientId,
    scope,
    additionalFields,
    headers
  } = config;
  const {
    clientSecret
  } = secrets;

  // `additionalFields` should be parseable, we do check API schema validation and in
  // action config validation step.
  let parsedAdditionalFields;
  try {
    parsedAdditionalFields = additionalFields ? JSON.parse(additionalFields) : undefined;
  } catch (error) {
    logger.error(`Connector ${connectorId}: error parsing additional fields`);
  }
  let accessToken;
  try {
    accessToken = await (0, _get_oauth_client_credentials_access_token.getOAuthClientCredentialsAccessToken)({
      connectorId,
      logger,
      configurationUtilities,
      oAuthScope: scope,
      credentials: {
        secrets: {
          clientSecret: clientSecret
        },
        config: {
          clientId: clientId,
          ...(parsedAdditionalFields ? {
            additionalFields: parsedAdditionalFields
          } : {})
        }
      },
      tokenUrl: accessTokenUrl,
      connectorTokenClient
    });
  } catch (error) {
    throw new Error(`Unable to retrieve/refresh the access token: ${error.message}`);
  }
  if (!accessToken) {
    throw new Error(`Unable to retrieve new access token`);
  }
  logger.debug(`Successfully retrieved access token`);
  const {
    onFulfilled,
    onRejected
  } = (0, _oauth2_delete_token_axios_interceptor.getOauth2DeleteTokenAxiosInterceptor)({
    connectorTokenClient,
    connectorId
  });
  const axiosInstance = _axios.default.create();
  axiosInstance.interceptors.response.use(onFulfilled, onRejected);
  const headersWithAuth = {
    ...headers,
    Authorization: accessToken
  };
  return {
    axiosInstance,
    headers: headersWithAuth,
    sslOverrides: {}
  };
};
const getDefaultAxiosConfig = async ({
  config,
  secrets
}) => {
  var _basicAuth$auth, _basicAuth$auth2;
  const {
    hasAuth,
    authType,
    verificationMode,
    ca,
    headers
  } = config;
  const axiosInstance = _axios.default.create();
  const {
    basicAuth,
    sslOverrides
  } = (0, _utils.buildConnectorAuth)({
    hasAuth,
    authType,
    secrets,
    verificationMode,
    ca
  });
  const mergedHeaders = (0, _lib.mergeConfigHeadersWithSecretHeaders)(headers, secrets.secretHeaders);
  const headersWithAuth = (0, _lib.combineHeadersWithBasicAuthHeader)({
    username: (_basicAuth$auth = basicAuth.auth) === null || _basicAuth$auth === void 0 ? void 0 : _basicAuth$auth.username,
    password: (_basicAuth$auth2 = basicAuth.auth) === null || _basicAuth$auth2 === void 0 ? void 0 : _basicAuth$auth2.password,
    headers: mergedHeaders
  });
  return {
    axiosInstance,
    headers: headersWithAuth,
    sslOverrides
  };
};
const getAxiosConfig = async ({
  config,
  secrets,
  connectorId,
  services,
  configurationUtilities,
  logger
}) => {
  let axiosConfig;
  try {
    if (config.authType === _constants.AuthType.OAuth2ClientCredentials) {
      axiosConfig = await getOAuth2AxiosConfig({
        connectorId,
        logger,
        configurationUtilities,
        services,
        config,
        secrets
      });
    } else {
      axiosConfig = await getDefaultAxiosConfig({
        config,
        secrets
      });
    }
    return [axiosConfig, null];
  } catch (error) {
    return [null, error];
  }
};
exports.getAxiosConfig = getAxiosConfig;