"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConnectorTypeId = void 0;
exports.executor = executor;
exports.getConnectorType = getConnectorType;
var _i18n = require("@kbn/i18n");
var _pipeable = require("fp-ts/pipeable");
var _Option = require("fp-ts/Option");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _common = require("@kbn/actions-plugin/common");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _common2 = require("@kbn/task-manager-plugin/common");
var _constants = require("../../../common/auth/constants");
var _http_response_retry_header = require("../lib/http_response_retry_header");
var _result_type = require("../lib/result_type");
var _schema = require("./schema");
var _schema2 = require("../../../common/auth/schema");
var _get_axios_config = require("./get_axios_config");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ConnectorTypeId = exports.ConnectorTypeId = '.webhook';
const userErrorCodes = [400, 404, 405, 406, 410, 411, 414, 428, 431];

// connector type definition
function getConnectorType() {
  return {
    id: ConnectorTypeId,
    minimumLicenseRequired: 'gold',
    name: _i18n.i18n.translate('xpack.stackConnectors.webhook.title', {
      defaultMessage: 'Webhook'
    }),
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId],
    validate: {
      config: {
        schema: _schema.ConfigSchema,
        customValidator: validateConnectorTypeConfig
      },
      secrets: {
        schema: _schema2.SecretConfigurationSchema
      },
      params: {
        schema: _schema.ParamsSchema
      }
    },
    renderParameterTemplates,
    executor
  };
}
function renderParameterTemplates(logger, params, variables) {
  if (!params.body) return params;
  return {
    body: (0, _mustache_renderer.renderMustacheString)(logger, params.body, variables, 'json')
  };
}
function validateUrl(configuredUrl) {
  try {
    new URL(configuredUrl);
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationErrorNoHostname', {
      defaultMessage: 'error validation webhook action config: unable to parse url: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
}
function ensureUriAllowed(configuredUrl, configurationUtilities) {
  try {
    configurationUtilities.ensureUriAllowed(configuredUrl);
  } catch (allowListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.configurationError', {
      defaultMessage: 'error validation webhook action config: {message}',
      values: {
        message: allowListError.message
      }
    }));
  }
}
function validateAuthType(configObject) {
  if (Boolean(configObject.authType) && !configObject.hasAuth) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.authConfigurationError', {
      defaultMessage: 'error validation webhook action config: authType must be null or undefined if hasAuth is false'
    }));
  }
}
function validateCertType(configObject, configurationUtilities) {
  if (configObject.certType === _constants.SSLCertType.PFX) {
    const webhookSettings = configurationUtilities.getWebhookSettings();
    if (!webhookSettings.ssl.pfx.enabled) {
      throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.pfxConfigurationError', {
        defaultMessage: 'error validation webhook action config: certType "{certType}" is disabled',
        values: {
          certType: _constants.SSLCertType.PFX
        }
      }));
    }
  }
}
function validateAdditionalFields(configObject) {
  if (configObject.additionalFields) {
    try {
      const parsedAdditionalFields = JSON.parse(configObject.additionalFields);
      if (typeof parsedAdditionalFields !== 'object' || Array.isArray(parsedAdditionalFields) || Object.keys(parsedAdditionalFields).length === 0) {
        throw new Error(_translations.ADDITIONAL_FIELD_CONFIG_ERROR);
      }
    } catch (e) {
      throw new Error(_translations.ADDITIONAL_FIELD_CONFIG_ERROR);
    }
  }
}
function validateOAuth2(configObject) {
  if (configObject.authType === _constants.AuthType.OAuth2ClientCredentials && (!configObject.accessTokenUrl || !configObject.clientId)) {
    const missingFields = [];
    if (!configObject.accessTokenUrl) {
      missingFields.push('Access Token URL (accessTokenUrl)');
    }
    if (!configObject.clientId) {
      missingFields.push('Client ID (clientId)');
    }
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.webhook.oauth2ConfigurationError', {
      defaultMessage: `error validation webhook action config: missing {missingItems} fields`,
      values: {
        missingItems: missingFields.join(', ')
      }
    }));
  }
}
function validateConnectorTypeConfig(configObject, validatorServices) {
  const {
    configurationUtilities
  } = validatorServices;
  const configuredUrl = configObject.url;
  validateUrl(configuredUrl);
  ensureUriAllowed(configuredUrl, configurationUtilities);
  validateAuthType(configObject);
  validateCertType(configObject, configurationUtilities);
  validateAdditionalFields(configObject);
  validateOAuth2(configObject);
}

// action executor
async function executor(execOptions) {
  const {
    actionId,
    config,
    params,
    configurationUtilities,
    logger,
    connectorUsageCollector,
    services
  } = execOptions;
  const {
    method,
    url
  } = config;
  const {
    body: data
  } = params;
  const [axiosConfig, axiosConfigError] = await (0, _get_axios_config.getAxiosConfig)({
    connectorId: actionId,
    services,
    config,
    secrets: execOptions.secrets,
    configurationUtilities,
    logger
  });
  if (axiosConfigError) {
    var _axiosConfigError$mes, _axiosConfigError$mes2;
    logger.error(`ConnectorId "${actionId}": error "${(_axiosConfigError$mes = axiosConfigError.message) !== null && _axiosConfigError$mes !== void 0 ? _axiosConfigError$mes : 'unknown error - couldnt load axios config'}"`);
    return errorResultRequestFailed(actionId, (_axiosConfigError$mes2 = axiosConfigError.message) !== null && _axiosConfigError$mes2 !== void 0 ? _axiosConfigError$mes2 : 'unknown error - couldnt load axios config');
  }
  const {
    axiosInstance,
    headers,
    sslOverrides
  } = axiosConfig;
  const result = await (0, _result_type.promiseResult)((0, _axios_utils.request)({
    axios: axiosInstance,
    method,
    url,
    logger,
    headers,
    data,
    configurationUtilities,
    sslOverrides,
    connectorUsageCollector
  }));
  if (result == null) {
    return errorResultUnexpectedNullResponse(actionId);
  }
  if ((0, _result_type.isOk)(result)) {
    const {
      value: {
        status,
        statusText
      }
    } = result;
    logger.debug(`response from webhook action "${actionId}": [HTTP ${status}] ${statusText}`);
    return successResult(actionId, data);
  } else {
    const {
      error
    } = result;
    if (error.response) {
      const {
        status,
        statusText,
        headers: responseHeaders,
        data: {
          message: responseMessage
        }
      } = error.response;
      const responseMessageAsSuffix = responseMessage ? `: ${responseMessage}` : '';
      const message = `[${status}] ${statusText}${responseMessageAsSuffix}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      // The request was made and the server responded with a status code
      // that falls out of the range of 2xx
      // special handling for 5xx
      if (status >= 500) {
        return retryResult(actionId, message);
      }

      // special handling for rate limiting
      if (status === 429) {
        return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(responseHeaders), (0, _Option.map)(retry => retryResultSeconds(actionId, message, retry)), (0, _Option.getOrElse)(() => retryResult(actionId, message)));
      }
      const errorResult = errorResultInvalid(actionId, message);
      if (userErrorCodes.includes(status)) {
        errorResult.errorSource = _common2.TaskErrorSource.USER;
      }
      return errorResult;
    } else if (error.code) {
      const message = `[${error.code}] ${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return errorResultRequestFailed(actionId, message);
    } else if (error.isAxiosError) {
      const message = `${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return errorResultRequestFailed(actionId, message);
    }
    logger.error(`error on ${actionId} webhook action: unexpected error`);
    return errorResultUnexpectedError(actionId);
  }
}

// Action Executor Result w/ internationalisation
function successResult(actionId, data) {
  return {
    status: 'ok',
    data,
    actionId
  };
}
function errorResultInvalid(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseErrorMessage', {
    defaultMessage: 'error calling webhook, invalid response'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage
  };
}
function errorResultRequestFailed(actionId, serviceMessage, errorSource) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.requestFailedErrorMessage', {
    defaultMessage: 'error calling webhook, request failed'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId,
    serviceMessage,
    errorSource
  };
}
function errorResultUnexpectedError(actionId) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.unreachableErrorMessage', {
    defaultMessage: 'error calling webhook, unexpected error'
  });
  return {
    status: 'error',
    message: errMessage,
    actionId
  };
}
function errorResultUnexpectedNullResponse(actionId) {
  const message = _i18n.i18n.translate('xpack.stackConnectors.webhook.unexpectedNullResponseErrorMessage', {
    defaultMessage: 'unexpected null response from webhook'
  });
  return {
    status: 'error',
    actionId,
    message
  };
}
function retryResult(actionId, serviceMessage) {
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseRetryLaterErrorMessage', {
    defaultMessage: 'error calling webhook, retry later'
  });
  return {
    status: 'error',
    message: errMessage,
    retry: true,
    actionId,
    serviceMessage
  };
}
function retryResultSeconds(actionId, serviceMessage, retryAfter) {
  const retryEpoch = Date.now() + retryAfter * 1000;
  const retry = new Date(retryEpoch);
  const retryString = retry.toISOString();
  const errMessage = _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidResponseRetryDateErrorMessage', {
    defaultMessage: 'error calling webhook, retry at {retryString}',
    values: {
      retryString
    }
  });
  return {
    status: 'error',
    message: errMessage,
    retry,
    actionId,
    serviceMessage
  };
}