"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.conditionToQueryDsl = conditionToQueryDsl;
var _conditions = require("../../types/conditions");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function conditionToClause(condition) {
  const operator = (0, _helpers.getFilterOperator)(condition);
  const value = (0, _helpers.getFilterValue)(condition);
  switch (operator) {
    case 'neq':
      return {
        bool: {
          must_not: {
            match: {
              [condition.field]: value
            }
          }
        }
      };
    case 'eq':
      return {
        match: {
          [condition.field]: value
        }
      };
    case 'exists':
      if (value === true) {
        return {
          exists: {
            field: condition.field
          }
        };
      } else if (value === false) {
        return {
          bool: {
            must_not: {
              exists: {
                field: condition.field
              }
            }
          }
        };
      }
    case 'gt':
      return {
        range: {
          [condition.field]: {
            gt: value
          }
        }
      };
    case 'gte':
      return {
        range: {
          [condition.field]: {
            gte: value
          }
        }
      };
    case 'lt':
      return {
        range: {
          [condition.field]: {
            lt: value
          }
        }
      };
    case 'lte':
      return {
        range: {
          [condition.field]: {
            lte: value
          }
        }
      };
    case 'contains':
      return {
        wildcard: {
          [condition.field]: {
            value: `*${value}*`,
            case_insensitive: true
          }
        }
      };
    case 'startsWith':
      return {
        prefix: {
          [condition.field]: `${value}*`
        }
      };
    case 'endsWith':
      return {
        wildcard: {
          [condition.field]: `*${value}`
        }
      };
    default:
      return {
        match_none: {}
      };
  }
}
function conditionToQueryDsl(condition) {
  if ((0, _conditions.isFilterCondition)(condition)) {
    return conditionToClause(condition);
  }
  if ((0, _conditions.isAndCondition)(condition)) {
    const and = condition.and.map(filter => conditionToQueryDsl(filter));
    return {
      bool: {
        must: and
      }
    };
  }
  if ((0, _conditions.isOrCondition)(condition)) {
    const or = condition.or.map(filter => conditionToQueryDsl(filter));
    return {
      bool: {
        should: or
      }
    };
  }
  if ((0, _conditions.isNotCondition)(condition)) {
    const not = conditionToQueryDsl(condition.not);
    return {
      bool: {
        must_not: not
      }
    };
  }
  if ((0, _conditions.isAlwaysCondition)(condition)) {
    return {
      match_all: {}
    };
  }
  return {
    match_none: {}
  };
}