"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dissectProcessorSchema = exports.dateProcessorSchema = exports.appendProcessorSchema = void 0;
exports.getProcessorType = getProcessorType;
exports.streamlangProcessorSchema = exports.renameProcessorSchema = exports.processorTypes = exports.manualIngestPipelineProcessorSchema = exports.isProcessWithOverrideOption = exports.isProcessWithIgnoreMissingOption = exports.isGrokProcessorDefinition = exports.isDissectProcessorDefinition = exports.isDateProcessorDefinition = exports.grokProcessorSchema = void 0;
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
var _conditions = require("../conditions");
var _fields = require("./fields");
var _manual_ingest_pipeline_processors = require("./manual_ingest_pipeline_processors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Base processor
 */

const processorBaseSchema = _zod.z.object({
  customIdentifier: _zod.z.optional(_zodHelpers.NonEmptyString),
  description: _zod.z.optional(_zod.z.string()),
  ignore_failure: _zod.z.optional(_zod.z.boolean())
});

/**
 * Base with where
 */

const processorBaseWithWhereSchema = processorBaseSchema.extend({
  where: _zod.z.optional(_conditions.conditionSchema)
});

/* Manual ingest pipeline processor */

// Not 100% accurate, but close enough for our use case to provide minimal safety
// without having to check all details

const manualIngestPipelineProcessorSchema = exports.manualIngestPipelineProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('manual_ingest_pipeline'),
  processors: _zod.z.array(_zod.z.record(_zod.z.enum(_manual_ingest_pipeline_processors.elasticsearchProcessorTypes), _zod.z.unknown())),
  tag: _zod.z.optional(_zod.z.string()),
  on_failure: _zod.z.optional(_zod.z.array(_zod.z.record(_zod.z.unknown())))
});

/**
 * Grok processor
 */

const grokProcessorSchema = exports.grokProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('grok'),
  from: _fields.StreamlangSourceField,
  patterns: _zod.z.array(_zodHelpers.NonEmptyString).nonempty(),
  ignore_missing: _zod.z.optional(_zod.z.boolean())
});

/**
 * Dissect processor
 */

const dissectProcessorSchema = exports.dissectProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('dissect'),
  from: _fields.StreamlangSourceField,
  pattern: _zodHelpers.NonEmptyString,
  append_separator: _zod.z.optional(_fields.StreamlangSeparator),
  ignore_missing: _zod.z.optional(_zod.z.boolean())
});

/**
 * Date processor
 */

const dateProcessorSchema = exports.dateProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('date'),
  from: _fields.StreamlangSourceField,
  to: _zod.z.optional(_fields.StreamlangTargetField),
  formats: _zod.z.array(_zodHelpers.NonEmptyString),
  output_format: _zod.z.optional(_zodHelpers.NonEmptyString)
});

/**
 * Rename processor
 */

const renameProcessorSchema = exports.renameProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('rename'),
  from: _fields.StreamlangSourceField,
  to: _fields.StreamlangTargetField,
  ignore_missing: _zod.z.optional(_zod.z.boolean()),
  override: _zod.z.optional(_zod.z.boolean())
});

/**
 * Set processor
 */

const setProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('set'),
  to: _fields.StreamlangTargetField,
  override: _zod.z.optional(_zod.z.boolean()),
  value: _zod.z.optional(_fields.NoMustacheValue),
  copy_from: _zod.z.optional(_fields.StreamlangSourceField)
}).refine(obj => obj.value && !obj.copy_from || !obj.value && obj.copy_from, {
  message: 'Set processor must have either value or copy_from, but not both.',
  path: ['value', 'copy_from']
});

/**
 * Append processor
 */

const appendProcessorSchema = exports.appendProcessorSchema = processorBaseWithWhereSchema.extend({
  action: _zod.z.literal('append'),
  to: _fields.StreamlangTargetField,
  value: _fields.NoMustacheArrayValues,
  // Rejects values like ["production", "{{{app}}}"]
  allow_duplicates: _zod.z.optional(_zod.z.boolean())
});
const streamlangProcessorSchema = exports.streamlangProcessorSchema = _zod.z.discriminatedUnion('action', [grokProcessorSchema, dissectProcessorSchema, dateProcessorSchema, renameProcessorSchema, setProcessorSchema.innerType(), appendProcessorSchema, manualIngestPipelineProcessorSchema]);
const isProcessWithOverrideOption = exports.isProcessWithOverrideOption = (0, _zodHelpers.createIsNarrowSchema)(processorBaseSchema, _zod.z.union([renameProcessorSchema, setProcessorSchema]));
const isProcessWithIgnoreMissingOption = exports.isProcessWithIgnoreMissingOption = (0, _zodHelpers.createIsNarrowSchema)(processorBaseSchema, _zod.z.union([renameProcessorSchema, grokProcessorSchema, dissectProcessorSchema]));
const isGrokProcessorDefinition = exports.isGrokProcessorDefinition = (0, _zodHelpers.createIsNarrowSchema)(streamlangProcessorSchema, grokProcessorSchema);
const isDissectProcessorDefinition = exports.isDissectProcessorDefinition = (0, _zodHelpers.createIsNarrowSchema)(streamlangProcessorSchema, dissectProcessorSchema);
const isDateProcessorDefinition = exports.isDateProcessorDefinition = (0, _zodHelpers.createIsNarrowSchema)(streamlangProcessorSchema, dateProcessorSchema);

/**
 * ProcessorType is the union of all possible 'action' values
 */

/**
 * Get all processor types as a string array (derived from the Zod schema)
 */
const processorTypes = exports.processorTypes = streamlangProcessorSchema._def.options.map(schema => schema.shape.action.value);

/**
 * Get the processor type (action) from a processor definition
 */
function getProcessorType(processor) {
  return processor.action;
}