"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.streamDetailRoute = exports.resolveIndexRoute = exports.listStreamsRoute = exports.internalCrudRoutes = void 0;
var _zod = require("@kbn/zod");
var _streamsSchema = require("@kbn/streams-schema");
var _process_async_in_chunks = require("../../../../utils/process_async_in_chunks");
var _constants = require("../../../../../common/constants");
var _create_server_route = require("../../../create_server_route");
var _stream_crud = require("../../../../lib/streams/stream_crud");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const listStreamsRoute = exports.listStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams',
  options: {
    access: 'internal'
  },
  params: _zod.z.object({}),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient,
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const streams = await streamsClient.listStreamsWithDataStreamExistence();
    const streamNames = streams.filter(({
      exists
    }) => exists).map(({
      stream
    }) => stream.name);
    let canReadFailureStore = true;
    const dataStreams = await (0, _process_async_in_chunks.processAsyncInChunks)(streamNames, async streamNamesChunk => {
      if (streamNamesChunk.length === 0) {
        return {
          data_streams: []
        };
      }
      const [{
        read_failure_store: readFailureStore
      }, dataStreamsChunk] = await Promise.all([streamsClient.getPrivileges(streamNamesChunk), scopedClusterClient.asCurrentUser.indices.getDataStream({
        name: streamNamesChunk
      })]);
      if (!readFailureStore) {
        canReadFailureStore = false;
      }
      return dataStreamsChunk;
    });
    const enrichedStreams = streams.reduce((acc, {
      stream
    }) => {
      if (_streamsSchema.Streams.GroupStream.Definition.is(stream)) {
        acc.push({
          stream
        });
        return acc;
      }
      const match = dataStreams.data_streams.find(dataStream => dataStream.name === stream.name);
      acc.push({
        stream,
        effective_lifecycle: (0, _stream_crud.getDataStreamLifecycle)(match !== null && match !== void 0 ? match : null),
        data_stream: match
      });
      return acc;
    }, []);
    return {
      streams: enrichedStreams,
      canReadFailureStore
    };
  }
});
const streamDetailRoute = exports.streamDetailRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/_details',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      start: _zod.z.string(),
      end: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const streamEntity = await streamsClient.getStream(params.path.name);
    if (_streamsSchema.Streams.GroupStream.Definition.is(streamEntity)) {
      return {
        details: {
          count: 0
        }
      };
    }

    // check doc count
    const docCountResponse = await scopedClusterClient.asCurrentUser.search({
      index: streamEntity.name,
      track_total_hits: true,
      ignore_unavailable: true,
      query: {
        range: {
          '@timestamp': {
            gte: params.query.start,
            lte: params.query.end
          }
        }
      },
      size: 0
    });
    const count = docCountResponse.hits.total.value;
    return {
      details: {
        count
      }
    };
  }
});
const resolveIndexRoute = exports.resolveIndexRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/_resolve_index',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    query: _zod.z.object({
      index: _zod.z.string()
    })
  }),
  handler: async ({
    request,
    params,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const response = (await scopedClusterClient.asCurrentUser.indices.get({
      index: params.query.index
    }))[params.query.index];
    const dataStream = response.data_stream;
    if (!dataStream) {
      return {};
    }
    const stream = await streamsClient.getStream(dataStream);
    return {
      stream
    };
  }
});
const internalCrudRoutes = exports.internalCrudRoutes = {
  ...listStreamsRoute,
  ...streamDetailRoute,
  ...resolveIndexRoute
};