"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.simulateProcessing = void 0;
var _elasticsearch = require("@elastic/elasticsearch");
var _objectUtils = require("@kbn/object-utils");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _streamlang = require("@kbn/streamlang");
var _hierarchy = require("@kbn/streams-schema/src/shared/hierarchy");
var _fields = require("@kbn/streams-schema/src/fields");
var _name = require("../../../../lib/streams/ingest_pipelines/name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/naming-convention */

// Narrow down the type to only successful processor results

/**
 * Detects and groups flattened geo_point fields (*.lat and *.lon) back into a single object.
 * Elasticsearch's geo_point mapper requires coordinates as a single object: { lat: number, lon: number }
 *
 * This function automatically detects any field pairs ending in .lat and .lon and groups them together.
 *
 * @param flattenedDoc - A flattened document with separate lat/lon fields
 * @returns Document with geo_point fields grouped as objects
 *
 * @example
 * Input: { "source.geo.location.lat": 41.9, "source.geo.location.lon": 42.0, "other": "value" }
 * Output: { "source.geo.location": { lat: 41.9, lon: 42.0 }, "other": "value" }
 */
const regroupGeoPointFields = flattenedDoc => {
  const result = {};
  const processedGeoFields = new Set();
  for (const [key, value] of Object.entries(flattenedDoc)) {
    // Check if this is a .lat field
    if (key.endsWith('.lat')) {
      const baseField = key.slice(0, -4); // Remove '.lat'
      const lonKey = `${baseField}.lon`;

      // Check if we have a corresponding .lon field with numeric values
      if (lonKey in flattenedDoc && !processedGeoFields.has(baseField) && typeof flattenedDoc[lonKey] === 'number' && typeof value === 'number') {
        // Group lat/lon into single object
        result[baseField] = {
          lat: value,
          lon: flattenedDoc[lonKey]
        };
        processedGeoFields.add(baseField);
        processedGeoFields.add(lonKey);
        continue;
      }
    }

    // Check if this field was already processed as part of a geo_point
    if (!processedGeoFields.has(key)) {
      result[key] = value;
    }
  }
  return result;
};
const simulateProcessing = async ({
  params,
  scopedClusterClient,
  streamsClient,
  fieldsMetadataClient
}) => {
  /* 0. Retrieve required data to prepare the simulation */
  const [stream, {
    indexState: streamIndexState,
    fieldCaps: streamIndexFieldCaps
  }] = await Promise.all([streamsClient.getStream(params.path.name), getStreamIndex(scopedClusterClient, streamsClient, params.path.name)]);

  /* 1. Prepare data for either simulation types (ingest, pipeline), prepare simulation body for the mandatory pipeline simulation */
  const simulationData = prepareSimulationData(params, stream);
  const pipelineSimulationBody = preparePipelineSimulationBody(simulationData);
  const ingestSimulationBody = prepareIngestSimulationBody(simulationData, stream, streamIndexState, params);
  /**
   * 2. Run both pipeline and ingest simulations in parallel.
   * - The pipeline simulation is used to extract the documents reports and the processor metrics. This always runs.
   * - The ingest simulation is used to fail fast on mapping failures. This runs only if `detected_fields` is provided.
   */
  const [pipelineSimulationResult, ingestSimulationResult] = await Promise.all([executePipelineSimulation(scopedClusterClient, pipelineSimulationBody), executeIngestSimulation(scopedClusterClient, ingestSimulationBody)]);

  /* 3. Fail fast on pipeline simulations errors and return the generic error response gracefully */
  if (pipelineSimulationResult.status === 'failure') {
    return prepareSimulationFailureResponse(pipelineSimulationResult.error);
  } else if (ingestSimulationResult.status === 'failure') {
    return prepareSimulationFailureResponse(ingestSimulationResult.error);
  }
  const streamFields = await getStreamFields(streamsClient, params.path.name);

  /* 4. Extract all the documents reports and processor metrics from the simulations */
  const {
    docReports,
    processorsMetrics
  } = computePipelineSimulationResult(pipelineSimulationResult.simulation, ingestSimulationResult.simulation, simulationData.docs, params.body.processing, _streamsSchema.Streams.WiredStream.Definition.is(stream), streamFields);

  /* 5. Extract valid detected fields with intelligent type suggestions from fieldsMetadataService */
  const detectedFields = await computeDetectedFields(processorsMetrics, params, streamFields, streamIndexFieldCaps, fieldsMetadataClient);

  /* 6. Derive general insights and process final response body */
  return prepareSimulationResponse(docReports, processorsMetrics, detectedFields);
};
exports.simulateProcessing = simulateProcessing;
const prepareSimulationDocs = (documents, streamName) => {
  return documents.map((doc, id) => ({
    _index: streamName,
    _id: id.toString(),
    _source: regroupGeoPointFields(doc)
  }));
};
const prepareSimulationProcessors = processing => {
  //
  /**
   * We want to simulate processors logic and collect data independently from the user config for simulation purposes.
   * 1. Force each processor to not ignore failures to collect all errors
   * 2. Append the error message to the `_errors` field on failure
   */
  const transpiledIngestPipelineProcessors = (0, _streamlang.transpileIngestPipeline)(processing, {
    ignoreMalformed: true,
    traceCustomIdentifiers: true
  }).processors;
  return transpiledIngestPipelineProcessors.map(processor => {
    const type = Object.keys(processor)[0];
    const processorConfig = processor[type]; // Safe to use any here due to type structure

    return {
      [type]: {
        ...processorConfig,
        ignore_failure: false,
        on_failure: [{
          append: {
            field: '_errors',
            value: {
              message: '{{{ _ingest.on_failure_message }}}',
              processor_id: processorConfig.tag,
              type: 'generic_processor_failure'
            }
          }
        }]
      }
    };
  });
};
const prepareSimulationData = (params, stream) => {
  const {
    body
  } = params;
  const {
    processing,
    documents
  } = body;
  const targetStreamName = _streamsSchema.Streams.WiredStream.Definition.is(stream) ? (0, _hierarchy.getRoot)(stream.name) : stream.name;
  return {
    docs: prepareSimulationDocs(documents, targetStreamName),
    processors: prepareSimulationProcessors(processing)
  };
};
const preparePipelineSimulationBody = simulationData => {
  const {
    docs,
    processors
  } = simulationData;
  return {
    docs,
    // @ts-expect-error field_access_pattern not supported by typing yet
    pipeline: {
      processors,
      field_access_pattern: 'flexible'
    },
    verbose: true
  };
};
const prepareIngestSimulationBody = (simulationData, stream, streamIndex, params) => {
  var _streamIndex$settings, _streamIndex$settings2;
  const {
    body
  } = params;
  const {
    detected_fields
  } = body;
  const {
    docs,
    processors
  } = simulationData;
  const defaultPipelineName = (_streamIndex$settings = streamIndex.settings) === null || _streamIndex$settings === void 0 ? void 0 : (_streamIndex$settings2 = _streamIndex$settings.index) === null || _streamIndex$settings2 === void 0 ? void 0 : _streamIndex$settings2.default_pipeline;
  const pipelineSubstitutions = {};
  if (defaultPipelineName) {
    pipelineSubstitutions[defaultPipelineName] = {
      processors,
      // @ts-expect-error field_access_pattern not supported by typing yet
      field_access_pattern: 'flexible'
    };
  }
  if (_streamsSchema.Streams.WiredStream.Definition.is(stream)) {
    // need to reroute from the root
    pipelineSubstitutions[(0, _name.getProcessingPipelineName)((0, _hierarchy.getRoot)(stream.name))] = {
      processors: [{
        reroute: {
          destination: stream.name
        }
      }]
    };
  }
  const simulationBody = {
    docs,
    pipeline_substitutions: pipelineSubstitutions,
    // Ideally we should not need to retrieve and merge the mappings from the stream index.
    // But the ingest simulation API does not validate correctly the mappings unless they are specified in the simulation body.
    // So we need to merge the mappings from the stream index with the detected fields.
    // This is a workaround until the ingest simulation API works as expected.
    ...(detected_fields && {
      mapping_addition: {
        properties: computeMappingProperties(detected_fields)
      }
    })
  };
  return simulationBody;
};

/**
 * When running a pipeline simulation, we want to fail fast on syntax failures, such as grok patterns.
 * If the simulation fails, we won't be able to extract the documents reports and the processor metrics.
 * In case any other error occurs, we delegate the error handling to currently in draft processor.
 */
const executePipelineSimulation = async (scopedClusterClient, simulationBody) => {
  try {
    const originalSimulation = await scopedClusterClient.asCurrentUser.ingest.simulate(simulationBody);
    const simulation = sanitiseSimulationResult(originalSimulation);
    return {
      status: 'success',
      simulation: simulation
    };
  } catch (error) {
    if (error instanceof _elasticsearch.errors.ResponseError) {
      var _error$body;
      const {
        processor_tag,
        reason
      } = (_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.error;
      return {
        status: 'failure',
        error: {
          message: reason,
          processor_id: processor_tag,
          type: 'generic_simulation_failure'
        }
      };
    }
    return {
      status: 'failure',
      error: {
        message: error.message,
        type: 'generic_simulation_failure'
      }
    };
  }
};

// When dealing with a manual_ingest_pipeline action it is possible to have nested pipelines in the configuration,
// as in, using the actual pipeline processor type. The problem is these results are a little bit different, e.g:
// {
//   "processor_type": "pipeline",
//   "status": "success",
//   "tag": "id5ded880-a555-11f0-94f6-45fc383ca38e",
//   "if": {
//     "condition": "ctx['data_stream.type'] == 'logs'",
//     "result": true
//   }
// },
// {
//   "processor_type": "set",
//   "status": "success",
//   "doc": {
//     "_index": "logs-synth-default",
//     "_version": "-3",
//     "_id": "99",
//     "_source": {
//       "host.name": test,
//     },
//   "_ingest": {
//     "pipeline": "network_subpipeline",
//     "timestamp": "2025-10-09T23:40:40.710774Z"
//   }
// }
// We use sanitiseSimulationResult and propagateProcessorResultsPipelineTags to
// propagate the pipeline processor tag (taken from the manual_ingest_pipeline action) to all nested
// pipeline processor results.
const sanitiseSimulationResult = simulationResult => {
  return {
    docs: simulationResult.docs.map(doc => {
      var _propagateProcessorRe;
      return {
        ...doc,
        processor_results: (_propagateProcessorRe = propagateProcessorResultsPipelineTags(doc.processor_results)) === null || _propagateProcessorRe === void 0 ? void 0 : _propagateProcessorRe.filter(result => {
          return result.processor_type !== 'pipeline';
        })
      };
    })
  };
};
function propagateProcessorResultsPipelineTags(processorResults) {
  if (!processorResults) return undefined;
  let lastPipelineTag;
  let applyTag = false;
  return processorResults.map(result => {
    var _result$doc, _result$doc$_ingest;
    // If this is a pipeline processor, store its tag and start applying
    if (result.processor_type === 'pipeline' && result.tag) {
      lastPipelineTag = result.tag;
      applyTag = true;
      return result;
    }

    // If 1. we should apply the tag 2. this result is not from the root simulated pipeline 3. has no tag set
    if (applyTag && !result.tag && ((_result$doc = result.doc) === null || _result$doc === void 0 ? void 0 : (_result$doc$_ingest = _result$doc._ingest) === null || _result$doc$_ingest === void 0 ? void 0 : _result$doc$_ingest.pipeline) !== '_simulate_pipeline') {
      // Apply the last pipeline tag
      return {
        ...result,
        tag: lastPipelineTag
      };
    }

    // If this result has its own tag, stop applying the pipeline tag
    if (result.tag) {
      applyTag = false;
    }
    return result;
  });
}
const executeIngestSimulation = async (scopedClusterClient, simulationBody) => {
  try {
    const simulation = await scopedClusterClient.asCurrentUser.simulate.ingest(simulationBody);
    return {
      status: 'success',
      simulation: simulation
    };
  } catch (error) {
    if (error instanceof _elasticsearch.errors.ResponseError) {
      var _error$body2;
      const {
        processor_tag,
        reason
      } = (_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.error;
      return {
        status: 'failure',
        error: {
          message: reason,
          processor_id: processor_tag,
          type: 'generic_simulation_failure'
        }
      };
    }
    return {
      status: 'failure',
      error: {
        message: error.message,
        type: 'generic_simulation_failure'
      }
    };
  }
};

/**
 * Computing simulation insights for each document and processor takes a few steps:
 * 1. Extract the last document source and the status of the simulation.
 * 2. Compute the diff between the sample document and the simulation document to detect fields changes.
 * 3. Track the detected fields and errors for each processor.
 *
 * To keep this process at the O(n) complexity, we iterate over the documents and processors only once.
 * This requires a closure on the processor metrics map to keep track of the processor state while iterating over the documents.
 */
const computePipelineSimulationResult = (pipelineSimulationResult, ingestSimulationResult, sampleDocs, processing, isWiredStream, streamFields) => {
  const transpiledProcessors = (0, _streamlang.transpileIngestPipeline)(processing, {
    ignoreMalformed: true,
    traceCustomIdentifiers: true
  }).processors;
  const processorsMap = initProcessorMetricsMap(transpiledProcessors);
  const forbiddenFields = Object.entries(streamFields).filter(([, {
    type
  }]) => type === 'system').map(([name]) => name);
  const docReports = pipelineSimulationResult.docs.map((pipelineDocResult, id) => {
    const ingestDocResult = ingestSimulationResult.docs[id];
    const ingestDocErrors = collectIngestDocumentErrors(ingestDocResult);
    const {
      errors,
      status,
      value
    } = getLastDoc(pipelineDocResult, sampleDocs[id]._source, ingestDocErrors);
    const diff = computeSimulationDocDiff(sampleDocs[id]._source, pipelineDocResult, isWiredStream, forbiddenFields);
    pipelineDocResult.processor_results.forEach(processor => {
      const procId = processor.tag;
      if (procId && isSkippedProcessor(processor)) {
        processorsMap[procId].skipped_rate++;
      }
    });
    diff.detected_fields.forEach(({
      processor_id,
      name
    }) => {
      processorsMap[processor_id].detected_fields.push(name);
    });
    errors.push(...diff.errors); // Add diffing errors to the document errors list, such as reserved fields
    errors.push(...ingestDocErrors); // Add ingestion errors to the document errors list, such as ignored_fields or mapping errors
    errors.forEach(error => {
      const procId = 'processor_id' in error && error.processor_id;
      if (procId && processorsMap[procId]) {
        processorsMap[procId].errors.push(error);
        processorsMap[procId].failed_rate++;
      }
    });
    return {
      detected_fields: diff.detected_fields,
      errors,
      status,
      value
    };
  });
  const processorsMetrics = extractProcessorMetrics({
    processorsMap,
    sampleSize: docReports.length
  });
  return {
    docReports,
    processorsMetrics
  };
};
const initProcessorMetricsMap = processors => {
  // Gather unique IDs because the manual ingest pipeline proccessor (for example) will share the same
  // ID across it's nested processors.
  const ids = new Set();
  for (const processor of processors) {
    const type = Object.keys(processor)[0];
    const config = processor[type];
    const tag = config.tag;
    if (typeof tag === 'string') {
      ids.add(tag);
    }
  }
  const uniqueIds = Array.from(ids);
  const processorMetricsEntries = uniqueIds.map(id => [id, {
    detected_fields: [],
    errors: [],
    failed_rate: 0,
    skipped_rate: 0,
    parsed_rate: 1
  }]);
  return Object.fromEntries(processorMetricsEntries);
};
const extractProcessorMetrics = ({
  processorsMap,
  sampleSize
}) => {
  return (0, _lodash.mapValues)(processorsMap, metrics => {
    const failureRate = metrics.failed_rate / sampleSize;
    const skippedRate = metrics.skipped_rate / sampleSize;
    const parsedRate = 1 - skippedRate - failureRate;
    const detected_fields = (0, _lodash.uniq)(metrics.detected_fields);
    const errors = (0, _lodash.uniqBy)(metrics.errors, error => error.message);
    return {
      detected_fields,
      errors,
      failed_rate: parseFloat(failureRate.toFixed(3)),
      skipped_rate: parseFloat(skippedRate.toFixed(3)),
      parsed_rate: parseFloat(parsedRate.toFixed(3))
    };
  });
};
const getDocumentStatus = (doc, ingestDocErrors) => {
  // If there is an ingestion mapping error, the document parsing should be considered failed
  if (ingestDocErrors.some(error => error.type === 'field_mapping_failure')) {
    return 'failed';
  }
  const processorResults = doc.processor_results;
  if (processorResults.every(isSkippedProcessor)) {
    return 'skipped';
  }
  if (processorResults.every(proc => isSuccessfulProcessor(proc) || isSkippedProcessor(proc))) {
    return 'parsed';
  }
  if (processorResults.some(isSuccessfulProcessor)) {
    return 'partially_parsed';
  }
  return 'failed';
};
const getLastDoc = (docResult, sample, ingestDocErrors) => {
  var _docResult$processor_, _docResult$processor_2, _docResult$processor_3;
  const status = getDocumentStatus(docResult, ingestDocErrors);
  const lastDocSource = (_docResult$processor_ = (_docResult$processor_2 = docResult.processor_results.filter(proc => !isSkippedProcessor(proc)).at(-1)) === null || _docResult$processor_2 === void 0 ? void 0 : (_docResult$processor_3 = _docResult$processor_2.doc) === null || _docResult$processor_3 === void 0 ? void 0 : _docResult$processor_3._source) !== null && _docResult$processor_ !== void 0 ? _docResult$processor_ : sample;
  if (status === 'parsed') {
    return {
      value: (0, _objectUtils.flattenObjectNestedLast)(lastDocSource),
      errors: [],
      status
    };
  } else {
    const {
      _errors = [],
      ...value
    } = lastDocSource;
    return {
      value: (0, _objectUtils.flattenObjectNestedLast)(value),
      errors: _errors,
      status
    };
  }
};

/**
 * To improve tracking down the errors and the fields detection to the individual processor,
 * this function computes the detected fields and the errors for each processor.
 */
const computeSimulationDocDiff = (base, docResult, isWiredStream, forbiddenFields) => {
  // Keep only the successful processors defined from the user, skipping the on_failure processors from the simulation
  const successfulProcessors = docResult.processor_results.filter(isSuccessfulProcessor);
  const comparisonDocs = [{
    processor_id: 'base',
    value: base
  }, ...successfulProcessors.map(proc => {
    return {
      processor_id: proc.tag,
      value: (0, _lodash.omit)(proc.doc._source, ['_errors'])
    };
  })];
  const diffResult = {
    detected_fields: [],
    errors: []
  };

  // Compare each document outcome with the previous one, flattening for standard comparison and detecting added/udpated fields.
  // When updated fields are detected compared to the original document, the processor is not additive to the documents, and an error is added to the diff result.
  while (comparisonDocs.length > 1) {
    const currentDoc = comparisonDocs.shift(); // Safe to use ! here since we check the length
    const nextDoc = comparisonDocs[0];
    const {
      added,
      updated
    } = (0, _objectUtils.calculateObjectDiff)((0, _objectUtils.flattenObjectNestedLast)(currentDoc.value), (0, _objectUtils.flattenObjectNestedLast)(nextDoc.value));
    const addedFields = Object.keys((0, _objectUtils.flattenObjectNestedLast)(added));
    const updatedFields = Object.keys((0, _objectUtils.flattenObjectNestedLast)(updated));

    // Sort list to have deterministic list of results
    const processorDetectedFields = [...addedFields, ...updatedFields].sort().map(name => ({
      processor_id: nextDoc.processor_id,
      name
    }));
    diffResult.detected_fields.push(...processorDetectedFields);
    if (isWiredStream) {
      const nonNamespacedFields = addedFields.filter(field => !(0, _streamsSchema.isNamespacedEcsField)(field));
      if (!(0, _lodash.isEmpty)(nonNamespacedFields)) {
        diffResult.errors.push({
          processor_id: nextDoc.processor_id,
          type: 'non_namespaced_fields_failure',
          message: `The fields generated by the processor do not match the streams recommended schema - put custom fields into attributes, body.structured or resource.attributes: [${nonNamespacedFields.join()}]`
        });
      }
    }
    if (forbiddenFields.some(field => updatedFields.includes(field))) {
      diffResult.errors.push({
        processor_id: nextDoc.processor_id,
        type: 'reserved_field_failure',
        message: `The processor is trying to update a reserved field [${forbiddenFields.join()}]`
      });
    }
  }
  return diffResult;
};
const collectIngestDocumentErrors = docResult => {
  var _docResult$doc2;
  const errors = [];
  if (isMappingFailure(docResult)) {
    var _docResult$doc, _docResult$doc$error;
    errors.push({
      type: 'field_mapping_failure',
      message: `Some field types might not be compatible with this document: ${(_docResult$doc = docResult.doc) === null || _docResult$doc === void 0 ? void 0 : (_docResult$doc$error = _docResult$doc.error) === null || _docResult$doc$error === void 0 ? void 0 : _docResult$doc$error.reason}`
    });
  }
  if ((_docResult$doc2 = docResult.doc) !== null && _docResult$doc2 !== void 0 && _docResult$doc2.ignored_fields) {
    errors.push({
      type: 'ignored_fields_failure',
      message: 'Some fields were ignored while simulating this document ingestion.',
      ignored_fields: docResult.doc.ignored_fields
    });
  }
  return errors;
};
const prepareSimulationResponse = async (docReports, processorsMetrics, detectedFields) => {
  const calculateRateByStatus = getRateCalculatorForDocs(docReports);
  const parsedRate = calculateRateByStatus('parsed');
  const partiallyParsedRate = calculateRateByStatus('partially_parsed');
  const skippedRate = calculateRateByStatus('skipped');
  const failureRate = calculateRateByStatus('failed');
  return {
    detected_fields: detectedFields,
    documents: docReports,
    processors_metrics: processorsMetrics,
    definition_error: undefined,
    documents_metrics: {
      failed_rate: parseFloat(failureRate.toFixed(3)),
      partially_parsed_rate: parseFloat(partiallyParsedRate.toFixed(3)),
      skipped_rate: parseFloat(skippedRate.toFixed(3)),
      parsed_rate: parseFloat(parsedRate.toFixed(3))
    }
  };
};
const prepareSimulationFailureResponse = error => {
  const failedBecauseNoSampleDocs = error.message.includes('must specify at least one document');
  return {
    detected_fields: [],
    documents: [],
    processors_metrics: {
      ...('processor_id' in error && error.processor_id && {
        [error.processor_id]: {
          detected_fields: [],
          errors: [error],
          failed_rate: 1,
          skipped_rate: 0,
          parsed_rate: 0
        }
      })
    },
    // failure to simulate is considered a definition error, which will be displayed prominently in the UI
    // The simulate API returns a generic error when no documents are provided, which is not useful in this context
    definition_error: !failedBecauseNoSampleDocs ? error : undefined,
    documents_metrics: {
      failed_rate: 1,
      partially_parsed_rate: 0,
      skipped_rate: 0,
      parsed_rate: 0
    }
  };
};
const getStreamIndex = async (scopedClusterClient, streamsClient, streamName) => {
  const dataStream = await streamsClient.getDataStream(streamName);
  const lastIndexRef = dataStream.indices.at(-1);
  if (!lastIndexRef) {
    throw new Error(`No writing index found for stream ${streamName}`);
  }
  const [lastIndex, lastIndexFieldCaps] = await Promise.all([scopedClusterClient.asCurrentUser.indices.get({
    index: lastIndexRef.index_name
  }), scopedClusterClient.asCurrentUser.fieldCaps({
    index: lastIndexRef.index_name,
    fields: '*'
  })]);
  return {
    indexState: lastIndex[lastIndexRef.index_name],
    fieldCaps: lastIndexFieldCaps.fields
  };
};
const getStreamFields = async (streamsClient, streamName) => {
  const [stream, ancestors] = await Promise.all([streamsClient.getStream(streamName), streamsClient.getAncestors(streamName)]);
  if (_streamsSchema.Streams.WiredStream.Definition.is(stream)) {
    return {
      ...stream.ingest.wired.fields,
      ...(0, _streamsSchema.getInheritedFieldsFromAncestors)(ancestors)
    };
  }
  if (_streamsSchema.Streams.ClassicStream.Definition.is(stream)) {
    return {
      ...stream.ingest.classic.field_overrides
    };
  }
  return {};
};

/**
 * In case new fields have been detected, we want to tell the user which ones are inherited and already mapped.
 */
const computeDetectedFields = async (processorsMetrics, params, streamFields, streamFieldCaps, fieldsMetadataClient) => {
  var _params$body$detected;
  const fields = Object.values(processorsMetrics).flatMap(metrics => metrics.detected_fields);
  const uniqueFields = (0, _lodash.uniq)(fields);

  // Short-circuit to avoid fetching streams fields if none is detected
  if ((0, _lodash.isEmpty)(uniqueFields)) {
    return [];
  }
  const confirmedValidDetectedFields = computeMappingProperties((_params$body$detected = params.body.detected_fields) !== null && _params$body$detected !== void 0 ? _params$body$detected : []);
  let fieldMetadataMap;
  try {
    fieldMetadataMap = (await fieldsMetadataClient.find({
      fieldNames: uniqueFields
    })).toPlain();
  } catch (error) {
    // Gracefully handle metadata service failures
    fieldMetadataMap = {};
  }
  return uniqueFields.map(name => {
    var _confirmedValidDetect;
    const existingField = streamFields[name];
    if (existingField) {
      return {
        name,
        ...existingField
      };
    }
    const existingFieldCaps = Object.keys(streamFieldCaps[name] || {});
    const esType = existingFieldCaps.length > 0 ? existingFieldCaps[0] : undefined;
    let suggestedType;
    let source;
    let description;
    const fieldMetadata = fieldMetadataMap[name];
    if (fieldMetadata && fieldMetadata.type && _fields.FIELD_DEFINITION_TYPES.includes(fieldMetadata.type)) {
      suggestedType = fieldMetadata.type;
      source = fieldMetadata.source;
      description = fieldMetadata.description;
    }
    return {
      name,
      type: (_confirmedValidDetect = confirmedValidDetectedFields[name]) === null || _confirmedValidDetect === void 0 ? void 0 : _confirmedValidDetect.type,
      esType,
      suggestedType,
      source,
      description
    };
  });
};
const getRateCalculatorForDocs = docs => status => {
  const matchCount = docs.reduce((rate, doc) => rate += doc.status === status ? 1 : 0, 0);
  return matchCount / docs.length;
};
const computeMappingProperties = detectedFields => {
  return Object.fromEntries(detectedFields.flatMap(({
    name,
    ...config
  }) => {
    if (config.type === 'system') {
      return [];
    }
    return [[name, {
      ...config,
      ignore_malformed: false
    }]];
  }));
};

/**
 * Guard helpers
 */
const isSuccessfulProcessor = processor => processor.status === 'success' && !!processor.tag;
const isSkippedProcessor = processor => processor.status === 'skipped';
const isMappingFailure = entry => {
  var _entry$doc, _entry$doc$error;
  return ((_entry$doc = entry.doc) === null || _entry$doc === void 0 ? void 0 : (_entry$doc$error = _entry$doc.error) === null || _entry$doc$error === void 0 ? void 0 : _entry$doc$error.type) === 'document_parsing_exception';
};