"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchEffectFactory = fetchEffectFactory;
exports.sendSuccessToast = exports.sendErrorToast = void 0;
var _effects = require("redux-saga/effects");
var _toast_title = require("../monitor_list/toast_title");
var _kibana_service = require("../../../../utils/kibana_service");
var _http_error = require("./http_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const sendSuccessToast = payload => {
  _kibana_service.kibanaService.toasts.addSuccess({
    title: (0, _toast_title.toastTitle)({
      title: payload.message,
      testAttribute: payload.testAttribute
    }),
    toastLifeTimeMs: payload.lifetimeMs
  });
};
exports.sendSuccessToast = sendSuccessToast;
const sendErrorToast = (payload, error) => {
  _kibana_service.kibanaService.toasts.addError(error, {
    ...payload.message,
    toastLifeTimeMs: payload.lifetimeMs
  });
};

/**
 * Factory function for a fetch effect. It expects three action creators,
 * one to call for a fetch, one to call for success, and one to handle failures.
 * @param fetch creates a fetch action
 * @param success creates a success action
 * @param fail creates a failure action
 * @param onSuccess
 * @param onFailure
 * @template T the action type expected by the fetch action
 * @template R the type that the API request should return on success
 * @template S the type of the success action
 * @template F the type of the failure action
 */
exports.sendErrorToast = sendErrorToast;
function fetchEffectFactory(fetch, success, fail, onSuccess, onFailure) {
  const showErrorToast = (error, action) => {
    const serializedError = (0, _http_error.serializeHttpFetchError)(error, action.payload);
    if (typeof onFailure === 'function') {
      onFailure === null || onFailure === void 0 ? void 0 : onFailure(error);
    } else if (typeof onFailure === 'string') {
      var _serializedError$body, _serializedError$body2;
      _kibana_service.kibanaService.coreSetup.notifications.toasts.addError({
        ...error,
        message: (_serializedError$body = (_serializedError$body2 = serializedError.body) === null || _serializedError$body2 === void 0 ? void 0 : _serializedError$body2.message) !== null && _serializedError$body !== void 0 ? _serializedError$body : error.message
      }, {
        title: onFailure
      });
    }
  };
  return function* (action) {
    try {
      const response = yield (0, _effects.call)(fetch, action.payload);
      if (response instanceof Error) {
        const error = response;
        // eslint-disable-next-line no-console
        console.error(error);
        const serializedError = (0, _http_error.serializeHttpFetchError)(error, action.payload);
        yield (0, _effects.put)(fail(serializedError));
        showErrorToast(error, action);
      } else {
        var _action$payload;
        yield (0, _effects.put)(success(response));
        const successMessage = (_action$payload = action.payload) === null || _action$payload === void 0 ? void 0 : _action$payload.success;
        if (successMessage !== null && successMessage !== void 0 && successMessage.message) {
          _kibana_service.kibanaService.toasts.addSuccess({
            title: (0, _toast_title.toastTitle)({
              title: successMessage.message,
              testAttribute: successMessage.testAttribute
            }),
            toastLifeTimeMs: successMessage.lifetimeMs
          });
        }
        if (typeof onSuccess === 'function') {
          onSuccess(response);
        } else if (onSuccess && typeof onSuccess === 'string') {
          _kibana_service.kibanaService.coreSetup.notifications.toasts.addSuccess(onSuccess);
        }
      }
    } catch (error) {
      var _action$payload2;
      // eslint-disable-next-line no-console
      console.error(error);
      const errorMessage = (_action$payload2 = action.payload) === null || _action$payload2 === void 0 ? void 0 : _action$payload2.error;
      if (errorMessage !== null && errorMessage !== void 0 && errorMessage.message) {
        _kibana_service.kibanaService.toasts.addError(error, {
          ...errorMessage.message,
          toastLifeTimeMs: errorMessage.lifetimeMs
        });
      }
      yield (0, _effects.put)(fail((0, _http_error.serializeHttpFetchError)(error, action.payload)));
      showErrorToast(error, action);
    }
  };
}