"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.invalidOriginError = exports.addSyntheticsMonitorRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _i18n = require("@kbn/i18n");
var _saved_objects = require("../../../common/types/saved_objects");
var _edit_monitor = require("./edit_monitor");
var _common_fields = require("../../synthetics_service/project_monitor/normalizers/common_fields");
var _add_monitor_api = require("./add_monitor/add_monitor_api");
var _constants = require("../../../common/constants");
var _monitor_validation = require("./monitor_validation");
var _saved_object_to_monitor = require("./formatters/saved_object_to_monitor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addSyntheticsMonitorRoute = () => ({
  method: 'POST',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS,
  validate: {},
  validation: {
    request: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        id: _configSchema.schema.maybe(_configSchema.schema.string()),
        preserve_namespace: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        gettingStarted: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        internal: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        })),
        // primarily used for testing purposes, to specify the type of saved object
        savedObjectType: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_saved_objects.syntheticsMonitorSavedObjectType), _configSchema.schema.literal(_saved_objects.legacySyntheticsMonitorTypeSingle)], {
          defaultValue: _saved_objects.syntheticsMonitorSavedObjectType
        }))
      })
    }
  },
  handler: async routeContext => {
    const {
      request,
      response,
      server,
      spaceId
    } = routeContext;
    // usually id is auto generated, but this is useful for testing
    const {
      id,
      internal,
      savedObjectType
    } = request.query;
    const addMonitorAPI = new _add_monitor_api.AddEditMonitorAPI(routeContext);
    const {
      locations,
      private_locations: privateLocations,
      ...monitor
    } = request.body;
    if (request.body.origin && request.body.origin !== 'ui') {
      return response.badRequest({
        body: {
          message: invalidOriginError(request.body.origin),
          attributes: {
            details: invalidOriginError(request.body.origin),
            payload: request.body
          }
        }
      });
    }
    try {
      const {
        errorMessage: unsupportedKeysErrors,
        formattedConfig
      } = (0, _monitor_validation.normalizeAPIConfig)(request.body);
      if (unsupportedKeysErrors) {
        return response.badRequest({
          body: {
            message: unsupportedKeysErrors,
            attributes: {
              details: unsupportedKeysErrors
            }
          }
        });
      }
      const monitorWithDefaults = await addMonitorAPI.normalizeMonitor(formattedConfig, request.body);
      const validationResult = (0, _monitor_validation.validateMonitor)(monitorWithDefaults, spaceId);
      if (!validationResult.valid || !validationResult.decodedMonitor) {
        const {
          reason: message,
          details
        } = validationResult;
        return response.badRequest({
          body: {
            message,
            attributes: {
              details,
              payload: monitor
            }
          }
        });
      }
      const normalizedMonitor = validationResult.decodedMonitor;

      // Parallelize permission and unique name validation
      const [err, nameError] = await Promise.all([(0, _edit_monitor.validatePermissions)(routeContext, normalizedMonitor.locations), addMonitorAPI.validateUniqueMonitorName(normalizedMonitor.name)]);
      if (err) {
        return response.forbidden({
          body: {
            message: err
          }
        });
      }
      if (nameError) {
        return response.badRequest({
          body: {
            message: nameError,
            attributes: {
              details: nameError
            }
          }
        });
      }
      const {
        errors,
        newMonitor
      } = await addMonitorAPI.syncNewMonitor({
        id,
        normalizedMonitor,
        savedObjectType
      });
      if (errors && errors.length > 0) {
        return {
          message: 'error pushing monitor to the service',
          attributes: {
            errors
          },
          id: newMonitor.id
        };
      }
      addMonitorAPI.initDefaultAlerts(newMonitor.attributes.name);
      addMonitorAPI.setupGettingStarted(newMonitor.id);
      return (0, _saved_object_to_monitor.mapSavedObjectToMonitor)({
        monitor: newMonitor,
        internal
      });
    } catch (error) {
      if (error instanceof _common_fields.InvalidLocationError || error instanceof _common_fields.InvalidScheduleError) {
        return response.badRequest({
          body: {
            message: error.message
          }
        });
      }
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(error)) {
        return response.forbidden({
          body: error
        });
      }
      server.logger.error('Unable to create synthetics monitor', {
        error
      });
      return response.customError({
        body: {
          message: error.message
        },
        statusCode: 500
      });
    }
  }
});
exports.addSyntheticsMonitorRoute = addSyntheticsMonitorRoute;
const invalidOriginError = origin => {
  return _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPublicOriginError', {
    defaultMessage: 'Unsupported origin type {origin}, only ui type is supported via API.',
    values: {
      origin
    }
  });
};
exports.invalidOriginError = invalidOriginError;