"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validatePermissions = exports.validateLocationPermissions = exports.syncEditedMonitor = exports.editSyntheticsMonitorRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _saved_objects = require("../../../common/types/saved_objects");
var _add_monitor = require("./add_monitor");
var _common_fields = require("../../synthetics_service/project_monitor/normalizers/common_fields");
var _add_monitor_api = require("./add_monitor/add_monitor_api");
var _add_monitor_project = require("./project_monitor/add_monitor_project");
var _get_private_locations = require("../../synthetics_service/get_private_locations");
var _saved_object_to_monitor = require("./formatters/saved_object_to_monitor");
var _runtime_types = require("../../../common/runtime_types");
var _constants = require("../../../common/constants");
var _monitor_validation = require("./monitor_validation");
var _service_errors = require("../synthetics_service/service_errors");
var _monitor_upgrade_sender = require("../telemetry/monitor_upgrade_sender");
var _secrets = require("../../synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Simplify return promise type and type it with runtime_types
const editSyntheticsMonitorRoute = () => ({
  method: 'PUT',
  path: _constants.SYNTHETICS_API_URLS.SYNTHETICS_MONITORS + '/{monitorId}',
  validate: {},
  validation: {
    request: {
      params: _configSchema.schema.object({
        monitorId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        internal: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        }))
      }),
      body: _configSchema.schema.any()
    }
  },
  handler: async routeContext => {
    const {
      request,
      response,
      spaceId,
      server,
      monitorConfigRepository
    } = routeContext;
    const {
      logger
    } = server;
    const monitor = request.body;
    const reqQuery = request.query;
    const {
      monitorId
    } = request.params;
    if (!monitor || typeof monitor !== 'object' || (0, _lodash.isEmpty)(monitor) || Array.isArray(monitor)) {
      return response.badRequest({
        body: {
          message: 'Monitor must be a non-empty object'
        }
      });
    }
    if (monitor.origin && monitor.origin !== 'ui') {
      return response.badRequest(getInvalidOriginError(monitor));
    }
    const editMonitorAPI = new _add_monitor_api.AddEditMonitorAPI(routeContext);
    if (monitor.name) {
      const nameError = await editMonitorAPI.validateUniqueMonitorName(monitor.name, monitorId);
      if (nameError) {
        return response.badRequest({
          body: {
            message: nameError,
            attributes: {
              details: nameError
            }
          }
        });
      }
    }
    try {
      /* Decrypting the previous monitor before editing ensures that all existing fields remain
       * on the object, even in flows where decryption does not take place, such as the enabled tab
       * on the monitor list table. We do not decrypt monitors in bulk for the monitor list table */
      const {
        decryptedMonitor: decryptedMonitorPrevMonitor,
        normalizedMonitor: previousMonitor
      } = await monitorConfigRepository.getDecrypted(monitorId, spaceId);
      const normalizedPreviousMonitor = previousMonitor.attributes;
      if (normalizedPreviousMonitor.origin !== 'ui' && !reqQuery.internal) {
        return response.badRequest(getInvalidOriginError(monitor));
      }
      let editedMonitor = (0, _saved_object_to_monitor.mergeSourceMonitor)(normalizedPreviousMonitor, monitor);
      editMonitorAPI.validateMonitorType(editedMonitor, normalizedPreviousMonitor);
      const {
        errorMessage: unsupportedKeysErrors,
        formattedConfig
      } = (0, _monitor_validation.normalizeAPIConfig)(editedMonitor);
      if (unsupportedKeysErrors) {
        return response.badRequest({
          body: {
            message: unsupportedKeysErrors,
            attributes: {
              details: unsupportedKeysErrors
            }
          }
        });
      }
      editedMonitor = await editMonitorAPI.normalizeMonitor(formattedConfig, monitor, previousMonitor.attributes.locations);
      const validationResult = (0, _monitor_validation.validateMonitor)(editedMonitor, spaceId);
      if (!validationResult.valid || !validationResult.decodedMonitor) {
        const {
          reason: message,
          details,
          payload
        } = validationResult;
        return response.badRequest({
          body: {
            message,
            attributes: {
              details,
              ...payload
            }
          }
        });
      }
      const err = await validatePermissions(routeContext, editedMonitor.locations);
      if (err) {
        return response.forbidden({
          body: {
            message: err
          }
        });
      }
      const monitorWithRevision = {
        ...validationResult.decodedMonitor,
        /* reset config hash to empty string. Ensures that the synthetics agent is able
         * to update project monitors on when next pushed after they are edited via the UI,
         * through the enable/disable monitor toggle */
        [_runtime_types.ConfigKey.CONFIG_HASH]: '',
        revision: (previousMonitor.attributes[_runtime_types.ConfigKey.REVISION] || 0) + 1
      };
      const {
        publicSyncErrors,
        failedPolicyUpdates,
        editedMonitor: editedMonitorSavedObject
      } = await syncEditedMonitor({
        routeContext,
        decryptedPreviousMonitor: decryptedMonitorPrevMonitor,
        normalizedMonitor: monitorWithRevision,
        spaceId
      });
      if (failedPolicyUpdates && failedPolicyUpdates.length > 0) {
        const hasError = failedPolicyUpdates.find(update => update.error);
        await rollbackUpdate({
          routeContext,
          configId: monitorId,
          attributes: decryptedMonitorPrevMonitor.attributes
        });
        throw hasError === null || hasError === void 0 ? void 0 : hasError.error;
      }

      // Return service sync errors in OK response
      if (publicSyncErrors && publicSyncErrors.length > 0) {
        return response.ok({
          body: {
            message: 'error pushing monitor to the service',
            attributes: {
              errors: publicSyncErrors
            }
          }
        });
      }
      return (0, _saved_object_to_monitor.mapSavedObjectToMonitor)({
        internal: reqQuery.internal,
        monitor: {
          ...editedMonitorSavedObject,
          created_at: previousMonitor.created_at
        }
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        return (0, _service_errors.getMonitorNotFoundResponse)(response, monitorId);
      }
      if (error instanceof _common_fields.InvalidLocationError || error instanceof _common_fields.InvalidScheduleError) {
        return response.badRequest({
          body: {
            message: error.message
          }
        });
      }
      if (error instanceof _monitor_validation.MonitorValidationError) {
        const {
          reason: message,
          details,
          payload
        } = error.result;
        return response.badRequest({
          body: {
            message,
            attributes: {
              details,
              ...payload
            }
          }
        });
      }
      logger.error(`Unable to update Synthetics monitor with id ${monitorId}, Error: ${error.message}`, {
        error
      });
      return response.customError({
        body: {
          message: error.message
        },
        statusCode: 500
      });
    }
  }
});
exports.editSyntheticsMonitorRoute = editSyntheticsMonitorRoute;
const rollbackUpdate = async ({
  routeContext,
  configId,
  attributes
}) => {
  const {
    savedObjectsClient,
    server
  } = routeContext;
  try {
    await savedObjectsClient.update(_saved_objects.syntheticsMonitorSavedObjectType, configId, attributes);
  } catch (error) {
    server.logger.error(`Unable to rollback edit for Synthetics monitor with id ${configId}, Error: ${error.message}`, {
      error
    });
  }
};
const syncEditedMonitor = async ({
  normalizedMonitor,
  decryptedPreviousMonitor,
  spaceId,
  routeContext
}) => {
  const {
    server,
    savedObjectsClient,
    syntheticsMonitorClient,
    monitorConfigRepository
  } = routeContext;
  try {
    const monitorWithId = {
      ...normalizedMonitor,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || decryptedPreviousMonitor.id,
      [_runtime_types.ConfigKey.CONFIG_ID]: decryptedPreviousMonitor.id,
      [_runtime_types.ConfigKey.KIBANA_SPACES]: normalizedMonitor[_runtime_types.ConfigKey.KIBANA_SPACES] || decryptedPreviousMonitor.namespaces
    };
    const formattedMonitor = (0, _secrets.formatSecrets)(monitorWithId);
    const editedSOPromise = monitorConfigRepository.update(decryptedPreviousMonitor.id, formattedMonitor, decryptedPreviousMonitor);
    const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(savedObjectsClient);
    const editSyncPromise = syntheticsMonitorClient.editMonitors([{
      monitor: monitorWithId,
      id: decryptedPreviousMonitor.id,
      decryptedPreviousMonitor
    }], allPrivateLocations, spaceId);
    const [editedMonitorSavedObject, {
      publicSyncErrors,
      failedPolicyUpdates
    }] = await Promise.all([editedSOPromise, editSyncPromise]);
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryUpdateEvent)(editedMonitorSavedObject, decryptedPreviousMonitor.updated_at, server.stackVersion, Boolean(normalizedMonitor[_runtime_types.ConfigKey.SOURCE_INLINE]), publicSyncErrors));
    return {
      failedPolicyUpdates,
      publicSyncErrors,
      editedMonitor: {
        ...editedMonitorSavedObject,
        attributes: {
          ...(editedMonitorSavedObject === null || editedMonitorSavedObject === void 0 ? void 0 : editedMonitorSavedObject.attributes),
          ...monitorWithId
        }
      }
    };
  } catch (e) {
    await rollbackUpdate({
      routeContext,
      configId: decryptedPreviousMonitor.id,
      attributes: decryptedPreviousMonitor.attributes
    });
    throw e;
  }
};
exports.syncEditedMonitor = syncEditedMonitor;
const validatePermissions = async (routeContext, monitorLocations) => {
  const hasPublicLocations = monitorLocations === null || monitorLocations === void 0 ? void 0 : monitorLocations.some(loc => loc.isServiceManaged);
  if (!hasPublicLocations) {
    return;
  }
  const {
    elasticManagedLocationsEnabled
  } = await validateLocationPermissions(routeContext);
  if (!elasticManagedLocationsEnabled) {
    return _add_monitor_project.ELASTIC_MANAGED_LOCATIONS_DISABLED;
  }
};
exports.validatePermissions = validatePermissions;
const validateLocationPermissions = async ({
  server,
  request
}) => {
  var _server$coreStart, _Boolean, _Boolean2;
  const uptimeFeature = await ((_server$coreStart = server.coreStart) === null || _server$coreStart === void 0 ? void 0 : _server$coreStart.capabilities.resolveCapabilities(request, {
    capabilityPath: 'uptime.*'
  }));
  const elasticManagedLocationsEnabled = (_Boolean = Boolean(uptimeFeature.uptime.elasticManagedLocationsEnabled)) !== null && _Boolean !== void 0 ? _Boolean : true;
  const canManagePrivateLocations = (_Boolean2 = Boolean(uptimeFeature.uptime.canManagePrivateLocations)) !== null && _Boolean2 !== void 0 ? _Boolean2 : true;
  return {
    canManagePrivateLocations,
    elasticManagedLocationsEnabled
  };
};
exports.validateLocationPermissions = validateLocationPermissions;
const getInvalidOriginError = monitor => {
  return {
    body: {
      message: (0, _add_monitor.invalidOriginError)(monitor.origin),
      attributes: {
        details: (0, _add_monitor.invalidOriginError)(monitor.origin),
        payload: monitor
      }
    }
  };
};