"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskManagerMetricsCollector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _mark_available_tasks_as_claimed = require("../queries/mark_available_tasks_as_claimed");
var _task_events = require("../task_events");
var _result_type = require("../lib/result_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_POLL_INTERVAL = 5000; // query every 5 seconds

class TaskManagerMetricsCollector {
  constructor({
    logger,
    store,
    pollInterval,
    taskTypes,
    excludedTypes
  }) {
    (0, _defineProperty2.default)(this, "store", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "pollInterval", void 0);
    (0, _defineProperty2.default)(this, "taskTypes", void 0);
    (0, _defineProperty2.default)(this, "excludedTypes", void 0);
    (0, _defineProperty2.default)(this, "running", false);
    // emit collected metrics
    (0, _defineProperty2.default)(this, "metrics$", new _rxjs.Subject());
    this.store = store;
    this.logger = logger;
    this.pollInterval = pollInterval !== null && pollInterval !== void 0 ? pollInterval : DEFAULT_POLL_INTERVAL;
    this.taskTypes = taskTypes;
    this.excludedTypes = excludedTypes;
    this.start();
  }
  get events() {
    return this.metrics$;
  }
  start() {
    if (!this.running) {
      this.running = true;
      this.runCollectionCycle().catch(() => {});
    }
  }
  async runCollectionCycle() {
    const start = Date.now();
    const searchedTypes = Array.from(this.taskTypes).filter(type => !this.excludedTypes.has(type));
    try {
      var _ref, _ref2, _ref3, _aggregations$overall;
      const results = await this.store.aggregate({
        size: 0,
        query: {
          bool: {
            filter: [{
              bool: {
                must: [(0, _mark_available_tasks_as_claimed.OneOfTaskTypes)('task.taskType', searchedTypes)],
                should: [_mark_available_tasks_as_claimed.IdleTaskWithExpiredRunAt, _mark_available_tasks_as_claimed.RunningOrClaimingTaskWithExpiredRetryAt],
                minimum_should_match: 1
              }
            }]
          }
        },
        runtime_mappings: {
          overdueBySeconds: {
            type: 'long',
            script: {
              source: `
                def taskStatus = doc['task.status'];
                def runAt = doc['task.runAt'];

                if (taskStatus.empty) {
                  emit(0);
                  return;
                }

                if(taskStatus.value == 'idle') {
                  emit((new Date().getTime() - runAt.value.getMillis()) / 1000);
                } else {
                  def retryAt = doc['task.retryAt'];
                  if(!retryAt.empty) {
                    emit((new Date().getTime() - retryAt.value.getMillis()) / 1000);
                  } else {
                    emit(0);
                  }
                }
              `
            }
          }
        },
        aggs: {
          overallOverdueByHistogram: {
            histogram: {
              field: 'overdueBySeconds',
              min_doc_count: 1,
              interval: 10
            }
          },
          byTaskType: {
            terms: {
              field: 'task.taskType',
              size: 500
            },
            aggs: {
              overdueByHistogram: {
                histogram: {
                  field: 'overdueBySeconds',
                  interval: 10
                }
              }
            }
          }
        }
      });
      const aggregations = (_ref = results === null || results === void 0 ? void 0 : results.aggregations) !== null && _ref !== void 0 ? _ref : {};
      const byTaskType = ((_ref2 = aggregations.byTaskType.buckets) !== null && _ref2 !== void 0 ? _ref2 : []).reduce((acc, bucket) => {
        var _bucket$overdueByHist, _bucket$overdueByHist2;
        // @ts-expect-error there's no way that buckets (array) matches `number`
        acc[bucket.key] = (_bucket$overdueByHist = bucket === null || bucket === void 0 ? void 0 : (_bucket$overdueByHist2 = bucket.overdueByHistogram) === null || _bucket$overdueByHist2 === void 0 ? void 0 : _bucket$overdueByHist2.buckets) !== null && _bucket$overdueByHist !== void 0 ? _bucket$overdueByHist : [];
        return acc;
      }, {});
      const metrics = {
        numOverdueTasks: {
          total: (_ref3 = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$overall = aggregations.overallOverdueByHistogram) === null || _aggregations$overall === void 0 ? void 0 : _aggregations$overall.buckets) !== null && _ref3 !== void 0 ? _ref3 : [],
          ...byTaskType
        }
      };
      this.metrics$.next((0, _task_events.asTaskManagerMetricEvent)((0, _result_type.asOk)(metrics)));
    } catch (e) {
      this.logger.debug(`Error querying for task manager metrics - ${e.message}`);
      // emit empty metrics so we don't have stale metrics
      this.metrics$.next((0, _task_events.asTaskManagerMetricEvent)((0, _result_type.asOk)({
        numOverdueTasks: {
          total: []
        }
      })));
    }
    if (this.running) {
      // Set the next runCycle call
      setTimeout(this.runCollectionCycle.bind(this), Math.max(this.pollInterval - (Date.now() - start), 0));
    }
  }
}
exports.TaskManagerMetricsCollector = TaskManagerMetricsCollector;