"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMigrationStatus = void 0;
var _react = require("react");
var _types = require("../../../../../../common/types");
var _types2 = require("../../../types");
var _readonly_state = require("./readonly_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLL_INTERVAL = 3000;
const getMigrationState = (migrationState, {
  migrationOp,
  warnings,
  hasRequiredPrivileges,
  meta: updatedMeta
}) => {
  const newMigrationState = {
    ...migrationState,
    // @ts-expect-error - resolutionType does non exist in all migration states.
    resolutionType: (migrationOp === null || migrationOp === void 0 ? void 0 : migrationOp.resolutionType) || migrationState.resolutionType,
    meta: updatedMeta || migrationState.meta,
    loadingState: _types2.LoadingState.Success
  };
  if (warnings) {
    newMigrationState.migrationWarnings = warnings;
  }
  if (hasRequiredPrivileges !== undefined) {
    newMigrationState.hasRequiredPrivileges = hasRequiredPrivileges;
  }
  if (migrationOp) {
    newMigrationState.status = migrationOp.status;
    if (migrationOp.status === _types.DataStreamMigrationStatus.notStarted) {
      return newMigrationState;
    }
    if (migrationOp.status === _types.DataStreamMigrationStatus.failed) {
      newMigrationState.errorMessage = migrationOp.errorMessage;
      return newMigrationState;
    }
    if (migrationOp.status === _types.DataStreamMigrationStatus.inProgress || migrationOp.status === _types.DataStreamMigrationStatus.completed) {
      newMigrationState.taskStatus = migrationOp.progressDetails;
      newMigrationState.taskPercComplete = migrationOp.taskPercComplete;
    }
    if (migrationState.cancelLoadingState === _types2.CancelLoadingState.Requested && migrationOp.status === _types.DataStreamMigrationStatus.inProgress) {
      newMigrationState.cancelLoadingState = _types2.CancelLoadingState.Loading;
    }
  }
  return newMigrationState;
};
const useMigrationStatus = ({
  dataStreamName,
  api
}) => {
  const [migrationState, setMigrationState] = (0, _react.useState)({
    loadingState: _types2.LoadingState.Loading,
    errorMessage: null,
    taskPercComplete: null,
    taskStatus: undefined,
    meta: null
  });
  const pollIntervalIdRef = (0, _react.useRef)(null);
  const readonlyState = (0, _react.useRef)(null);
  const isMounted = (0, _react.useRef)(false);
  const clearPollInterval = (0, _react.useCallback)(() => {
    if (pollIntervalIdRef.current) {
      clearTimeout(pollIntervalIdRef.current);
      pollIntervalIdRef.current = null;
    }
  }, []);
  const pollingFunction = (0, _react.useCallback)(async resolutionType => {
    clearPollInterval();
    try {
      if (resolutionType === 'readonly' && !readonlyState.current) {
        return;
      }
      let data = null;
      let error = null;
      if (resolutionType === 'readonly') {
        if (!readonlyState.current) {
          throw new Error('Readonly state not initialized');
        }
        const {
          value
        } = await readonlyState.current.next();
        data = value;
      } else {
        const results = await api.getDataStreamMigrationStatus(dataStreamName);
        data = results.data;
        error = results.error;
      }
      if (error) {
        setMigrationState(prevValue => {
          return {
            ...prevValue,
            loadingState: _types2.LoadingState.Error,
            errorMessage: error.message.toString(),
            status: _types.DataStreamMigrationStatus.fetchFailed
          };
        });
        return;
      }
      if (!data) {
        return;
      }
      setMigrationState(prevValue => {
        return getMigrationState(prevValue, data);
      });
      if (data.migrationOp && data.migrationOp.status === _types.DataStreamMigrationStatus.inProgress) {
        // Only keep polling if it exists and is in progress.
        pollIntervalIdRef.current = setTimeout(() => pollingFunction(migrationState.resolutionType), POLL_INTERVAL);
      }
    } catch (error) {
      setMigrationState(prevValue => {
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Error,
          errorMessage: error.message.toString(),
          status: _types.DataStreamMigrationStatus.fetchFailed
        };
      });
    }
  }, [clearPollInterval, api, dataStreamName, migrationState.resolutionType]);
  const updateStatus = (0, _react.useCallback)(async () => {
    return pollingFunction(migrationState.resolutionType);
  }, [pollingFunction, migrationState.resolutionType]);
  const startReindex = (0, _react.useCallback)(async () => {
    setMigrationState(prevValue => {
      return {
        ...prevValue,
        status: _types.DataStreamMigrationStatus.inProgress,
        taskPercComplete: null,
        errorMessage: null,
        cancelLoadingState: undefined
      };
    });
    if (migrationState.status === _types.DataStreamMigrationStatus.failed) {
      try {
        await api.cancelDataStreamReindexTask(dataStreamName);
      } catch (_) {
        // if the task has already failed, attempt to cancel the task
        // before attempting to start the reindexing again.
      }
    }
    const {
      data: migrationOp,
      error
    } = await api.startDataStreamReindexTask(dataStreamName);
    if (error) {
      setMigrationState(prevValue => {
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Error,
          errorMessage: error.message.toString(),
          status: _types.DataStreamMigrationStatus.failed
        };
      });
      return;
    }
    setMigrationState(prevValue => {
      return getMigrationState(prevValue, {
        migrationOp,
        meta: prevValue.meta
      });
    });
    updateStatus();
  }, [api, dataStreamName, updateStatus, migrationState.status]);
  const loadDataStreamMetadata = (0, _react.useCallback)(async () => {
    try {
      const {
        data,
        error
      } = await api.getDataStreamMetadata(dataStreamName);
      if (error) {
        throw error;
      }
      setMigrationState(prevValue => {
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Success,
          meta: data || null
        };
      });
    } catch (error) {
      setMigrationState(prevValue => {
        // if state is completed, we don't need to update the meta
        if (prevValue.status === _types.DataStreamMigrationStatus.completed) {
          return prevValue;
        }
        return {
          ...prevValue,
          loadingState: _types2.LoadingState.Error,
          errorMessage: error.message.toString(),
          status: _types.DataStreamMigrationStatus.failed
        };
      });
    }
  }, [api, dataStreamName]);
  const cancelReindex = (0, _react.useCallback)(async () => {
    setMigrationState(prevValue => {
      return {
        ...prevValue,
        cancelLoadingState: _types2.CancelLoadingState.Requested
      };
    });
    try {
      const {
        error
      } = await api.cancelDataStreamReindexTask(dataStreamName);
      if (error) {
        throw error;
      }
      setMigrationState(prevValue => {
        return {
          ...prevValue,
          cancelLoadingState: _types2.CancelLoadingState.Success,
          status: _types.DataStreamMigrationStatus.cancelled
        };
      });
    } catch (error) {
      setMigrationState(prevValue => {
        return {
          ...prevValue,
          cancelLoadingState: _types2.CancelLoadingState.Error
        };
      });
    }
  }, [api, dataStreamName]);
  const startReadonly = (0, _react.useCallback)(async () => {
    /**
     * Here we jsut mark the status as in progress for the polling function
     * to start executing the reindexing.
     */
    setMigrationState(prevValue => {
      return {
        ...prevValue,
        resolutionType: 'readonly',
        status: _types.DataStreamMigrationStatus.inProgress,
        taskPercComplete: null
      };
    });
    readonlyState.current = (0, _readonly_state.readOnlyExecute)(dataStreamName, migrationState.meta, api);
    pollingFunction('readonly');
  }, [api, dataStreamName, migrationState, pollingFunction]);
  const cancelReadonly = (0, _react.useCallback)(async () => {
    readonlyState.current = null;
    setMigrationState(prevValue => {
      return {
        ...prevValue,
        resolutionType: undefined,
        cancelLoadingState: _types2.CancelLoadingState.Success,
        status: _types.DataStreamMigrationStatus.cancelled
      };
    });
  }, []);
  const initMigration = (0, _react.useCallback)(resolutionType => {
    setMigrationState(prevValue => {
      return {
        ...prevValue,
        resolutionType,
        status: _types.DataStreamMigrationStatus.notStarted
      };
    });
  }, []);
  (0, _react.useEffect)(() => {
    updateStatus();
  }, [updateStatus]);
  (0, _react.useEffect)(() => {
    isMounted.current = true;
    return () => {
      isMounted.current = false;

      // Clean up on unmount.
      clearPollInterval();
    };
  }, [clearPollInterval]);
  return {
    migrationState,
    loadDataStreamMetadata,
    initMigration,
    updateStatus,
    startReindex,
    cancelReindex,
    startReadonly,
    cancelReadonly
  };
};
exports.useMigrationStatus = useMigrationStatus;