"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSnapshotState = void 0;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLL_INTERVAL_MS = 1000;
const useSnapshotState = ({
  jobId,
  snapshotId,
  api
}) => {
  const [requestError, setRequestError] = (0, _react.useState)(undefined);
  const [snapshotState, setSnapshotState] = (0, _react.useState)({
    status: 'idle',
    jobId,
    snapshotId
  });
  const pollIntervalIdRef = (0, _react.useRef)(null);
  const isMounted = (0, _react.useRef)(false);
  const clearPollInterval = (0, _react.useCallback)(() => {
    if (pollIntervalIdRef.current) {
      clearTimeout(pollIntervalIdRef.current);
      pollIntervalIdRef.current = null;
    }
  }, []);
  const updateSnapshotStatus = (0, _react.useCallback)(async () => {
    clearPollInterval();
    const {
      data,
      error: updateStatusError
    } = await api.getMlSnapshotUpgradeStatus({
      jobId,
      snapshotId
    });
    if (updateStatusError) {
      setSnapshotState({
        snapshotId,
        jobId,
        action: 'upgrade',
        status: 'error'
      });
      setRequestError(updateStatusError);
      return;
    }
    setSnapshotState({
      ...data,
      action: 'upgrade'
    });

    // Only keep polling if it exists and is in progress.
    if ((data === null || data === void 0 ? void 0 : data.status) === 'in_progress') {
      pollIntervalIdRef.current = setTimeout(updateSnapshotStatus, POLL_INTERVAL_MS);
    }
  }, [api, clearPollInterval, jobId, snapshotId]);
  const upgradeSnapshot = (0, _react.useCallback)(async () => {
    setSnapshotState({
      snapshotId,
      jobId,
      action: 'upgrade',
      status: 'in_progress'
    });
    const {
      data,
      error: upgradeError
    } = await api.upgradeMlSnapshot({
      jobId,
      snapshotId
    });
    if (upgradeError) {
      setRequestError(upgradeError);
      setSnapshotState({
        snapshotId,
        jobId,
        action: 'upgrade',
        status: 'error'
      });
      return;
    }
    setSnapshotState({
      ...data,
      action: 'upgrade'
    });
    updateSnapshotStatus();
  }, [api, jobId, snapshotId, updateSnapshotStatus]);
  const deleteSnapshot = (0, _react.useCallback)(async () => {
    setSnapshotState({
      snapshotId,
      jobId,
      action: 'delete',
      status: 'in_progress'
    });
    const {
      error: deleteError
    } = await api.deleteMlSnapshot({
      snapshotId,
      jobId
    });
    if (deleteError) {
      setRequestError(deleteError);
      setSnapshotState({
        snapshotId,
        jobId,
        action: 'delete',
        status: 'error'
      });
      return;
    }
    setSnapshotState({
      snapshotId,
      jobId,
      action: 'delete',
      status: 'complete'
    });
  }, [api, jobId, snapshotId]);
  (0, _react.useEffect)(() => {
    isMounted.current = true;
    return () => {
      isMounted.current = false;

      // Clean up on unmount.
      clearPollInterval();
    };
  }, [clearPollInterval]);
  return {
    snapshotState: {
      ...snapshotState,
      error: requestError
    },
    upgradeSnapshot,
    updateSnapshotStatus,
    deleteSnapshot
  };
};
exports.useSnapshotState = useSnapshotState;