"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createParentPipelineAggregationColumn = exports.convertParentPipelineAggToColumns = exports.convertMetricAggregationToColumn = exports.convertMetricAggregationColumnWithoutSpecialParams = exports.computeParentPipelineColumns = void 0;
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _metrics = require("../metrics");
var _column = require("./column");
var _formula = require("./formula");
var _moving_average = require("./moving_average");
var _percentile = require("./percentile");
var _percentile_rank = require("./percentile_rank");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SUPPORTED_METRICS_AGGS_WITHOUT_PARAMS = [_convert_to_lens.Operations.AVERAGE, _convert_to_lens.Operations.COUNT, _convert_to_lens.Operations.UNIQUE_COUNT, _convert_to_lens.Operations.MAX, _convert_to_lens.Operations.MIN, _convert_to_lens.Operations.SUM, _convert_to_lens.Operations.STANDARD_DEVIATION, _convert_to_lens.Operations.COUNTER_RATE];
const SUPPORTED_METRIC_AGGS = [...SUPPORTED_METRICS_AGGS_WITHOUT_PARAMS, _convert_to_lens.Operations.LAST_VALUE, _convert_to_lens.Operations.PERCENTILE, _convert_to_lens.Operations.PERCENTILE_RANK, _convert_to_lens.Operations.COUNTER_RATE];
const isSupportedAggregation = agg => {
  return SUPPORTED_METRIC_AGGS.includes(agg);
};
const isSupportedAggregationWithoutParams = agg => {
  return SUPPORTED_METRICS_AGGS_WITHOUT_PARAMS.includes(agg);
};
const convertMetricAggregationColumnWithoutSpecialParams = (aggregation, {
  series,
  metrics,
  dataView
}, additionalArgs) => {
  if (!isSupportedAggregationWithoutParams(aggregation.name)) {
    return null;
  }
  const metric = metrics[metrics.length - 1];
  const sourceField = aggregation.isFieldRequired && metric.field ? metric.field : 'document';
  const field = dataView.getFieldByName(sourceField);
  if (!field && aggregation.isFieldRequired) {
    return null;
  }
  return {
    operationType: aggregation.name,
    sourceField,
    ...(0, _column.createColumn)(series, metric, field, additionalArgs),
    // dataType has to be number in Lens to inherit the formatter
    ...(sourceField === 'document' ? {
      dataType: 'number'
    } : {}),
    params: {
      ...(0, _column.getFormat)(series)
    }
  };
};
exports.convertMetricAggregationColumnWithoutSpecialParams = convertMetricAggregationColumnWithoutSpecialParams;
const convertMetricAggregationToColumn = (aggregation, {
  series,
  metric,
  dataView
}, {
  metaValue,
  reducedTimeRange
} = {}) => {
  var _metric$field;
  if (!isSupportedAggregation(aggregation.name)) {
    return null;
  }
  const field = dataView.getFieldByName((_metric$field = metric.field) !== null && _metric$field !== void 0 ? _metric$field : 'document');
  if (!field && aggregation.isFieldRequired) {
    return null;
  }
  if (aggregation.name === _convert_to_lens.Operations.PERCENTILE) {
    return (0, _percentile.convertToPercentileColumn)(metaValue, {
      series,
      metric,
      dataView
    }, {
      reducedTimeRange
    });
  }
  if (aggregation.name === _convert_to_lens.Operations.PERCENTILE_RANK) {
    return (0, _percentile_rank.convertToPercentileRankColumn)(metaValue === null || metaValue === void 0 ? void 0 : metaValue.toString(), series, metric, dataView, {
      reducedTimeRange
    });
  }
  if (aggregation.name === _convert_to_lens.Operations.LAST_VALUE) {
    return null;
  }
  return convertMetricAggregationColumnWithoutSpecialParams(aggregation, {
    series,
    metrics: [metric],
    dataView
  }, {
    reducedTimeRange
  });
};
exports.convertMetricAggregationToColumn = convertMetricAggregationToColumn;
const computeParentPipelineColumns = (aggregation, {
  series,
  metric,
  dataView
}, subFunctionMetric, pipelineAgg, {
  metaValue,
  reducedTimeRange
} = {}) => {
  const agg = _metrics.SUPPORTED_METRICS[metric.type];
  if (!agg) {
    return null;
  }
  const aggFormula = (0, _metrics.getFormulaFromMetric)(agg);
  if (subFunctionMetric.type === 'filter_ratio') {
    const script = (0, _metrics.getFilterRatioFormula)(subFunctionMetric, {
      reducedTimeRange,
      timeShift: series.offset_time
    });
    if (!script) {
      return null;
    }
    const formula = `${aggFormula}(${script})`;
    return (0, _formula.createFormulaColumn)(formula, {
      series,
      metric,
      dataView
    });
  }
  const metricAggregationColumn = convertMetricAggregationToColumn(pipelineAgg, {
    series,
    metric: subFunctionMetric,
    dataView
  }, {
    metaValue,
    reducedTimeRange
  });
  if (!metricAggregationColumn) {
    return null;
  }
  return [metricAggregationColumn, createParentPipelineAggregationColumn(aggregation, {
    series,
    metric,
    dataView
  }, [metricAggregationColumn.columnId])];
};
exports.computeParentPipelineColumns = computeParentPipelineColumns;
const convertMovingAvgOrDerivativeToColumns = (aggregation, metric, {
  series,
  metrics,
  dataView
}, reducedTimeRange) => {
  var _metric$field$split, _metric$field2, _subMetricField$split;
  //  percentile value is derived from the field Id. It has the format xxx-xxx-xxx-xxx[percentile]
  const [fieldId, meta] = (_metric$field$split = metric === null || metric === void 0 ? void 0 : (_metric$field2 = metric.field) === null || _metric$field2 === void 0 ? void 0 : _metric$field2.split('[')) !== null && _metric$field$split !== void 0 ? _metric$field$split : [];
  const subFunctionMetric = metrics.find(({
    id
  }) => id === fieldId);
  if (!subFunctionMetric || subFunctionMetric.type === 'static') {
    return null;
  }
  const pipelineAgg = _metrics.SUPPORTED_METRICS[subFunctionMetric.type];
  if (!pipelineAgg) {
    return null;
  }
  const metaValue = Number(meta === null || meta === void 0 ? void 0 : meta.replace(/\]/g, ''));
  const subMetricField = subFunctionMetric.field;
  const [nestedFieldId, _] = (_subMetricField$split = subMetricField === null || subMetricField === void 0 ? void 0 : subMetricField.split('[')) !== null && _subMetricField$split !== void 0 ? _subMetricField$split : [];
  // support nested aggs with formula
  const additionalSubFunction = metrics.find(({
    id
  }) => id === nestedFieldId);
  if (additionalSubFunction || pipelineAgg.name === 'counter_rate' || subFunctionMetric.type === 'variance') {
    const formula = (0, _metrics.getPipelineSeriesFormula)(metric, metrics, subFunctionMetric, {
      metaValue,
      reducedTimeRange,
      timeShift: series.offset_time
    });
    if (!formula) {
      return null;
    }
    return (0, _formula.createFormulaColumn)(formula, {
      series,
      metric,
      dataView
    });
  } else {
    const agg = _metrics.SUPPORTED_METRICS[aggregation];
    if (!agg) {
      return null;
    }
    return computeParentPipelineColumns(agg.name, {
      series,
      metric,
      dataView
    }, subFunctionMetric, pipelineAgg, {
      metaValue,
      reducedTimeRange
    });
  }
};
const convertParentPipelineAggToColumns = ({
  series,
  metrics,
  dataView
}, reducedTimeRange) => {
  const currentMetric = metrics[metrics.length - 1];
  if (currentMetric.type === 'moving_average' || currentMetric.type === 'derivative') {
    return convertMovingAvgOrDerivativeToColumns(currentMetric.type, currentMetric, {
      series,
      metrics,
      dataView
    }, reducedTimeRange);
  }
  return null;
};
exports.convertParentPipelineAggToColumns = convertParentPipelineAggToColumns;
const createParentPipelineAggregationColumn = (aggregation, {
  series,
  metric,
  dataView
}, references = []) => {
  const params = aggregation === 'moving_average' ? (0, _moving_average.convertToMovingAverageParams)(metric) : (0, _column.getFormat)(series);
  return {
    operationType: aggregation,
    references,
    ...(0, _column.createColumn)(series, metric, undefined, {
      timeShift: series.offset_time
    }),
    params
  };
};
exports.createParentPipelineAggregationColumn = createParentPipelineAggregationColumn;