"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AxisConfig = void 0;
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _lodash = _interopRequireDefault(require("lodash"));
var _d = _interopRequireDefault(require("d3"));
var _scale_modes = require("./scale_modes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const defaults = {
  show: true,
  type: 'value',
  elSelector: '.visAxis__column--{pos} .axis-div',
  position: 'left',
  scale: {
    type: 'linear',
    expandLastBucket: true,
    inverted: false,
    defaultYExtents: null,
    boundsMargin: 0,
    setYExtents: null,
    min: null,
    max: null,
    mode: _scale_modes.SCALE_MODES.NORMAL
  },
  style: {
    color: '#ddd',
    lineWidth: '1px',
    opacity: 1,
    tickColor: '#ddd',
    tickWidth: '1px',
    tickLength: '6px',
    rangePadding: 0.1,
    rangeOuterPadding: 0
  },
  labels: {
    axisFormatter: null,
    show: true,
    rotate: 0,
    rotateAnchor: 'center',
    filter: false,
    color: '#ddd',
    font: '"Open Sans", "Lato", "Helvetica Neue", Helvetica, Arial, sans-serif',
    fontSize: '8pt',
    truncate: 100
  },
  title: {
    text: '',
    elSelector: '.visAxis__column--{pos} .axis-div'
  },
  padForLabels: 0
};
const padForLabelsX = 40;
const padForLabelsY = 15;
const categoryDefaults = {
  type: 'category',
  position: 'bottom'
};
const valueDefaults = {
  labels: {
    axisFormatter: _d.default.format('n')
  }
};
const horizontalDefaults = {
  labels: {
    rotate: 0,
    rotateAnchor: 'end',
    filter: true,
    truncate: 0
  }
};
const verticalDefaults = {
  labels: {
    rotateAnchor: 'middle'
  }
};
class AxisConfig {
  constructor(chartConfig, axisConfigArgs) {
    const isCategoryType = axisConfigArgs.type === 'category';
    const typeDefaults = isCategoryType ? _lodash.default.cloneDeep(categoryDefaults) : _lodash.default.cloneDeep(valueDefaults);
    // _.defaultsDeep mutates axisConfigArgs nested values so we clone it first
    const axisConfigArgsClone = _lodash.default.cloneDeep(axisConfigArgs);
    const isCategoryAxis = axisConfigArgsClone.type === 'category';
    const isHorizontal = axisConfigArgsClone.position && ['top', 'bottom'].includes(axisConfigArgsClone.position);
    _lodash.default.merge(typeDefaults, isHorizontal || isCategoryAxis ? horizontalDefaults : verticalDefaults);
    this._values = _lodash.default.defaultsDeep({}, axisConfigArgsClone, typeDefaults, defaults);
    this._values.elSelector = this._values.elSelector.replace('{pos}', this._values.position);
    this._values.rootEl = chartConfig.get('el');
    this.data = chartConfig.data;
    if (isCategoryAxis) {
      if (!this._values.values) {
        this.values = this.data.xValues(chartConfig.get('orderBucketsBySum', false));
        this.ordered = this.data.get('ordered');
      } else {
        this.values = this._values.values;
      }
      if (!this._values.labels.axisFormatter) {
        this._values.labels.axisFormatter = this.data.data.xAxisFormatter || this.data.get('xAxisFormatter');
      }
    }
    if (this.get('type') === 'value') {
      const isWiggleOrSilhouette = this.get('scale.mode') === _scale_modes.SCALE_MODES.WIGGLE || this.get('scale.mode') === _scale_modes.SCALE_MODES.SILHOUETTE;
      // if show was not explicitly set and wiggle or silhouette option was checked
      if (isWiggleOrSilhouette) {
        this._values.scale.defaultYExtents = false;
        if (!axisConfigArgs.show) {
          this._values.show = false;
          this._values.title.show = true;
        }
      }
      if (this.isLogScale()) {
        this._values.labels.filter = true;
      }
    }
    if (axisConfigArgs.title == null || axisConfigArgs.title.text == null) {
      const label = isCategoryAxis ? 'xAxisLabel' : 'yAxisLabel';
      this.set('title.text', this.data.get(label));
    }

    // horizontal axis with ordinal scale should have labels rotated (so we can fit more)
    // unless explicitly overridden by user
    if (this.isHorizontal() && this.isOrdinal()) {
      this._values.labels.filter = _lodash.default.get(axisConfigArgs, 'labels.filter', false);
      this._values.labels.rotate = _lodash.default.get(axisConfigArgs, 'labels.rotate', 90);
      this._values.labels.truncate = _lodash.default.get(axisConfigArgs, 'labels.truncate', 100);
    }
    if (this.get('type') === 'category' && !this.isHorizontal()) {
      this._values.scale.inverted = _lodash.default.get(axisConfigArgs, 'scale.inverted', true);
    }
    if (chartConfig.get('labels.show', false) && !isCategoryAxis) {
      this._values.padForLabels = isHorizontal ? padForLabelsX : padForLabelsY;
    }
    let offset;
    let stacked = true;
    switch (this.get('scale.mode')) {
      case _scale_modes.SCALE_MODES.NORMAL:
        offset = 'zero';
        stacked = false;
        break;
      case _scale_modes.SCALE_MODES.GROUPED:
        offset = 'group';
        stacked = false;
        break;
      case _scale_modes.SCALE_MODES.PERCENTAGE:
        offset = function expand(data) {
          // taken from https://github.com/d3/d3/blob/v3.5.6/src/layout/stack.js#L193
          // fixed to support zeros
          const n = data.length;
          const m = data[0].length;
          const y0 = [];
          for (let j = 0; j < m; ++j) {
            let o = 0;
            for (let i = 0; i < n; i++) {
              o += data[i][j][1];
            }
            if (o) {
              for (let i = 0; i < n; i++) {
                data[i][j][1] /= o;
              }
            }
          }
          for (let j = 0; j < m; ++j) {
            y0[j] = 0;
          }
          return y0;
        };
        break;
      default:
        offset = this.get('scale.mode');
    }
    this.set('scale.offset', _lodash.default.get(axisConfigArgs, 'scale.offset', offset));
    /* axis.scale.stacked means that axis stacking function should be run */
    this.set('scale.stacked', stacked);
  }
  get(property, defaults) {
    if (typeof defaults === 'undefined' && !_lodash.default.has(this._values, property)) {
      throw new Error(`Accessing invalid config property: ${property}`);
    }
    const val = _lodash.default.get(this._values, property, defaults);
    if (val == null && defaults != null) return defaults;
    return val;
  }
  set(property, value) {
    return (0, _saferLodashSet.set)(this._values, property, value);
  }
  isHorizontal() {
    return this._values.position === 'top' || this._values.position === 'bottom';
  }
  isOrdinal() {
    return !!this.values && !this.isTimeDomain();
  }
  isTimeDomain() {
    return this.ordered && this.ordered.date;
  }
  isPercentage() {
    return this._values.scale.mode === _scale_modes.SCALE_MODES.PERCENTAGE;
  }
  isUserDefined() {
    return this._values.scale.setYExtents;
  }
  isYExtents() {
    return this._values.scale.defaultYExtents;
  }
  isLogScale() {
    return this.getScaleType() === 'log';
  }
  getScaleType() {
    return this._values.scale.type;
  }
}
exports.AxisConfig = AxisConfig;