"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.serializeState = exports.deserializeState = exports.deserializeSavedObjectState = void 0;
var _lodash = require("lodash");
var _services = require("../services");
var _saved_visualize_utils = require("../utils/saved_visualize_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const deserializeState = async state => {
  if (!state.rawState) return {
    serializedVis: {
      data: {}
    }
  };
  if (state.rawState.savedObjectId) {
    return await deserializeSavedObjectState(state.rawState);
  }
  const {
    savedVis,
    ...rest
  } = state.rawState;
  return {
    ...rest,
    serializedVis: savedVis
  };
};
exports.deserializeState = deserializeState;
const deserializeSavedObjectState = async ({
  savedObjectId,
  enhancements,
  uiState,
  timeRange,
  title: embeddableTitle,
  description: embeddableDescription,
  hidePanelTitles
}) => {
  // Load a saved visualization from the library
  const {
    title,
    description,
    visState,
    searchSource,
    searchSourceFields,
    savedSearchId,
    savedSearchRefName,
    uiStateJSON,
    ...savedObjectProperties
  } = await (0, _saved_visualize_utils.getSavedVisualization)({
    dataViews: (0, _services.getDataViews)(),
    search: (0, _services.getSearch)(),
    savedObjectsTagging: (0, _services.getSavedObjectTagging)().getTaggingApi(),
    spaces: (0, _services.getSpaces)(),
    i18n: (0, _services.getI18n)(),
    overlays: (0, _services.getOverlays)(),
    analytics: (0, _services.getAnalytics)(),
    theme: (0, _services.getTheme)(),
    userProfile: (0, _services.getUserProfile)()
  }, savedObjectId);
  const panelTitle = embeddableTitle !== null && embeddableTitle !== void 0 ? embeddableTitle : title;
  const panelDescription = embeddableDescription !== null && embeddableDescription !== void 0 ? embeddableDescription : description;
  return {
    serializedVis: {
      title,
      type: visState.type,
      params: visState.params,
      uiState: uiState !== null && uiState !== void 0 ? uiState : uiStateJSON ? JSON.parse(uiStateJSON) : {},
      data: {
        aggs: visState.aggs,
        searchSource: searchSource !== null && searchSource !== void 0 ? searchSource : searchSourceFields,
        savedSearchId
      }
    },
    title: panelTitle,
    description: panelDescription,
    hidePanelTitles,
    savedObjectId,
    savedObjectProperties,
    linkedToLibrary: true,
    ...(timeRange ? {
      timeRange
    } : {}),
    ...(enhancements ? {
      enhancements
    } : {})
  };
};
exports.deserializeSavedObjectState = deserializeSavedObjectState;
const serializeState = ({
  serializedVis,
  // Serialize the vis before passing it to this function for easier testing
  titles,
  id,
  savedObjectProperties,
  linkedToLibrary,
  getDynamicActionsState,
  timeRange
}) => {
  const dynamicActionsState = getDynamicActionsState ? getDynamicActionsState() : {};
  // save by reference
  if (linkedToLibrary && id) {
    return {
      rawState: {
        ...(titles ? titles : {}),
        ...dynamicActionsState,
        ...(!(0, _lodash.isEmpty)(serializedVis.uiState) ? {
          uiState: serializedVis.uiState
        } : {}),
        ...(timeRange ? {
          timeRange
        } : {}),
        savedObjectId: id
      },
      references: []
    };
  }
  return {
    rawState: {
      ...(titles ? titles : {}),
      ...savedObjectProperties,
      ...dynamicActionsState,
      ...(timeRange ? {
        timeRange
      } : {}),
      savedVis: {
        ...serializedVis,
        id
      }
    },
    references: []
  };
};
exports.serializeState = serializeState;