"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVisualizationInstanceFromInput = exports.getVisualizationInstance = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/kibana-utils-plugin/common");
var _vis_async = require("../../vis_async");
var _saved_visualize_utils = require("../../utils/saved_visualize_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isErrorRelatedToRuntimeFields(error) {
  const originalError = error.original || error;
  return originalError instanceof _common.SavedFieldNotFound || originalError instanceof _common.SavedFieldTypeInvalidForAgg;
}
const createVisualizeEmbeddableAndLinkSavedSearch = async (vis, visualizeServices) => {
  const {
    data,
    createVisEmbeddableFromObject,
    savedSearch: savedSearchApi
  } = visualizeServices;
  let savedSearch;
  if (vis.data.savedSearchId) {
    try {
      savedSearch = vis.data.savedSearchId ? await savedSearchApi.get(vis.data.savedSearchId) : await savedSearchApi.getNew();
    } catch (e) {
      // skip this catch block
    }
  }
  const embeddableHandler = await createVisEmbeddableFromObject(vis, {
    id: '',
    timeRange: data.query.timefilter.timefilter.getTime(),
    filters: data.query.filterManager.getFilters(),
    searchSessionId: data.search.session.getSessionId(),
    renderMode: 'edit'
  });
  embeddableHandler.getOutput$().subscribe(output => {
    if (output.error && !isErrorRelatedToRuntimeFields(output.error)) {
      data.search.showError(output.error.original || output.error);
    }
  });
  return {
    savedSearch,
    embeddableHandler
  };
};
const getVisualizationInstanceFromInput = async (visualizeServices, input) => {
  var _input$title, _input$description, _input$timeRange;
  const {
    data,
    spaces,
    savedObjectsTagging,
    ...startServices
  } = visualizeServices;
  const visState = input.savedVis;

  /**
   * A saved vis is needed even in by value mode to support 'save to library' which converts the 'by value'
   * state of the visualization, into a new saved object.
   */
  const savedVis = await (0, _saved_visualize_utils.getSavedVisualization)({
    search: data.search,
    spaces,
    savedObjectsTagging,
    ...startServices
  });
  if (visState.uiState && Object.keys(visState.uiState).length !== 0) {
    savedVis.uiStateJSON = JSON.stringify(visState.uiState);
  }
  let vis = await (0, _vis_async.createVisAsync)(visState.type, (0, _lodash.cloneDeep)(visState));
  if (vis.type.setup) {
    try {
      vis = await vis.type.setup(vis);
    } catch {
      // skip this catch block
    }
  }
  const {
    embeddableHandler,
    savedSearch
  } = await createVisualizeEmbeddableAndLinkSavedSearch(vis, visualizeServices);
  return {
    vis,
    savedVis,
    embeddableHandler,
    savedSearch,
    panelTitle: (_input$title = input === null || input === void 0 ? void 0 : input.title) !== null && _input$title !== void 0 ? _input$title : '',
    panelDescription: (_input$description = input === null || input === void 0 ? void 0 : input.description) !== null && _input$description !== void 0 ? _input$description : '',
    panelTimeRange: (_input$timeRange = input === null || input === void 0 ? void 0 : input.timeRange) !== null && _input$timeRange !== void 0 ? _input$timeRange : undefined
  };
};
exports.getVisualizationInstanceFromInput = getVisualizationInstanceFromInput;
const getVisualizationInstance = async (visualizeServices, opts) => {
  const {
    data,
    spaces,
    savedObjectsTagging,
    ...startServices
  } = visualizeServices;
  const savedVis = await (0, _saved_visualize_utils.getSavedVisualization)({
    search: data.search,
    spaces,
    savedObjectsTagging,
    ...startServices
  }, opts);
  if (typeof opts !== 'string') {
    savedVis.searchSourceFields = {
      index: opts === null || opts === void 0 ? void 0 : opts.indexPattern
    };
  }
  const serializedVis = (0, _saved_visualize_utils.convertToSerializedVis)(savedVis);
  let vis = await (0, _vis_async.createVisAsync)(serializedVis.type, serializedVis);
  if (vis.type.setup) {
    try {
      vis = await vis.type.setup(vis);
    } catch {
      // skip this catch block
    }
  }
  const {
    embeddableHandler,
    savedSearch
  } = await createVisualizeEmbeddableAndLinkSavedSearch(vis, visualizeServices);
  return {
    vis,
    embeddableHandler,
    savedSearch,
    savedVis
  };
};
exports.getVisualizationInstance = getVisualizationInstance;