"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StepExecutionRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class StepExecutionRepository {
  constructor(esClient) {
    (0, _defineProperty2.default)(this, "indexName", _common.WORKFLOWS_STEP_EXECUTIONS_INDEX);
    this.esClient = esClient;
  }

  /**
   * Searches for step executions by workflow execution ID.
   *
   * @param executionId - The ID of the workflow execution to search for step executions.
   * @returns A promise that resolves to an array of step executions associated with the given execution ID.
   */
  async searchStepExecutionsByExecutionId(executionId) {
    const response = await this.esClient.search({
      index: this.indexName,
      query: {
        match: {
          workflowRunId: executionId
        }
      },
      sort: 'startedAt:desc'
    });
    return response.hits.hits.map(hit => hit._source);
  }

  /**
   * Creates a new step execution document in Elasticsearch.
   *
   * @param stepExecution - A partial object representing the workflow step execution to be indexed.
   * @returns A promise that resolves when the document has been successfully indexed.
   */
  async createStepExecution(stepExecution) {
    if (!stepExecution.id) {
      throw new Error('Step execution ID is required for creation');
    }
    await this.esClient.index({
      index: this.indexName,
      id: stepExecution.id,
      refresh: true,
      document: stepExecution
    });
  }

  /**
   * Updates a single workflow step execution in the repository.
   *
   * @param stepExecution - A partial object representing the workflow step execution to update.
   * @returns A promise that resolves when the update operation is complete.
   */
  updateStepExecution(stepExecution) {
    return this.updateStepExecutions([stepExecution]);
  }

  /**
   * Updates multiple step executions in Elasticsearch.
   *
   * This method takes an array of partial `EsWorkflowStepExecution` objects,
   * validates that each has an `id`, and performs a bulk update operation
   * in Elasticsearch for all provided step executions.
   *
   * @param stepExecutions - An array of partial step execution objects to update.
   * Each object must include an `id` property.
   * @throws {Error} If any step execution does not have an `id`.
   * @returns A promise that resolves when the bulk update operation completes.
   */
  async updateStepExecutions(stepExecutions) {
    var _this$esClient;
    stepExecutions.forEach(stepExecution => {
      if (!stepExecution.id) {
        throw new Error('Step execution ID is required for update');
      }
    });
    await ((_this$esClient = this.esClient) === null || _this$esClient === void 0 ? void 0 : _this$esClient.bulk({
      refresh: true,
      index: this.indexName,
      body: stepExecutions.flatMap(stepExecution => [{
        update: {
          _id: stepExecution.id
        }
      }, {
        doc: stepExecution
      }])
    }));
  }
}
exports.StepExecutionRepository = StepExecutionRepository;