"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowExecutionRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowExecutionRepository {
  constructor(esClient) {
    (0, _defineProperty2.default)(this, "indexName", _common.WORKFLOWS_EXECUTIONS_INDEX);
    this.esClient = esClient;
  }

  /**
   * Retrieves a workflow execution by its ID from Elasticsearch.
   *
   * @param workflowExecutionId - The ID of the workflow execution to retrieve.
   * @param spaceId - The ID of the space associated with the workflow execution.
   * @returns A promise that resolves to the workflow execution document, or null if not found.
   */
  async getWorkflowExecutionById(workflowExecutionId, spaceId) {
    const response = await this.esClient.search({
      index: this.indexName,
      query: {
        bool: {
          filter: [{
            term: {
              id: workflowExecutionId
            }
          }, {
            term: {
              spaceId
            }
          }]
        }
      },
      size: 1
    });
    if (response.hits.hits.length === 0) {
      return null;
    }
    return response.hits.hits[0]._source;
  }

  /**
   * Creates a new workflow execution document in Elasticsearch.
   *
   * @param workflowExecution - A partial object representing the workflow execution to be created.
   * @returns A promise that resolves when the workflow execution has been indexed.
   */
  async createWorkflowExecution(workflowExecution) {
    if (!workflowExecution.id) {
      throw new Error('Workflow execution ID is required for creation');
    }
    await this.esClient.index({
      index: this.indexName,
      id: workflowExecution.id,
      refresh: true,
      document: workflowExecution
    });
  }

  /**
   * Partially updates an existing workflow execution in Elasticsearch.
   *
   * This method requires the `id` property to be present in the `workflowExecution` object.
   * If the `id` is missing, an error is thrown.
   * The update operation is performed using the Elasticsearch client, and the document is refreshed after the update.
   *
   * @param workflowExecution - A partial object representing the workflow execution to update. Must include the `id` property.
   * @throws {Error} If the `id` property is not provided in the `workflowExecution` object.
   * @returns A promise that resolves when the update operation is complete.
   */
  async updateWorkflowExecution(workflowExecution) {
    if (!workflowExecution.id) {
      throw new Error('Workflow execution ID is required for update');
    }
    await this.esClient.update({
      index: this.indexName,
      id: workflowExecution.id,
      refresh: true,
      doc: workflowExecution
    });
  }
}
exports.WorkflowExecutionRepository = WorkflowExecutionRepository;