"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AtomicStepImpl = void 0;
var _connector_step = require("../connector_step");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Implements the execution logic for an atomic workflow step.
 *
 * `AtomicStepImpl` is responsible for running a single atomic step within a workflow.
 * It delegates the execution to a `ConnectorStepImpl`, passing the necessary configuration,
 * context manager, connector executor, and workflow state.
 *
 * @remarks
 * This class is typically used internally by the workflow execution engine to process
 * atomic nodes in the workflow graph.
 *
 * @param node - The atomic graph node containing step configuration.
 * @param contextManager - Manages workflow context and state.
 * @param connectorExecutor - Executes connector operations for the step.
 * @param workflowState - Manages the runtime state of workflow execution.
 */
class AtomicStepImpl {
  constructor(node, contextManager, connectorExecutor, workflowState, workflowLogger) {
    this.node = node;
    this.contextManager = contextManager;
    this.connectorExecutor = connectorExecutor;
    this.workflowState = workflowState;
    this.workflowLogger = workflowLogger;
  }
  async run() {
    // This class should decide what action to take based on action type
    // like connector, logger, http call, etc.
    // for now it only calls ConnectorStepImpl
    await new _connector_step.ConnectorStepImpl(this.node.configuration, this.contextManager, this.connectorExecutor, this.workflowState, this.workflowLogger).run();
  }
}
exports.AtomicStepImpl = AtomicStepImpl;