"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticsearchActionStepImpl = void 0;
var _workflows = require("@kbn/workflows");
var _node_implementation = require("./node_implementation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Extend BaseStep for elasticsearch-specific properties

class ElasticsearchActionStepImpl extends _node_implementation.BaseAtomicNodeImplementation {
  constructor(step, contextManager, workflowRuntime, workflowLogger) {
    super(step, contextManager, undefined, workflowRuntime);
    this.workflowLogger = workflowLogger;
  }
  getInput() {
    var _configuration;
    // Get current context for templating
    const context = this.contextManager.getContext();
    // Render inputs from 'with' - support both direct step.with and step.configuration.with
    const stepWith = this.step.with || ((_configuration = this.step.configuration) === null || _configuration === void 0 ? void 0 : _configuration.with) || {};
    return this.renderObjectTemplate(stepWith, context);
  }

  /**
   * Recursively render the object template.
   * @param obj - The object to render.
   * @param context - The context to use for rendering.
   * @returns The rendered object.
   */
  renderObjectTemplate(obj, context) {
    if (Array.isArray(obj)) {
      return obj.map(item => this.renderObjectTemplate(item, context));
    }
    if (obj && typeof obj === 'object') {
      return Object.entries(obj).reduce((acc, [key, value]) => {
        acc[key] = this.renderObjectTemplate(value, context);
        return acc;
      }, {});
    }
    if (typeof obj === 'string') {
      return this.templatingEngine.render(obj, context);
    }
    return obj;
  }
  async _run(withInputs) {
    try {
      var _configuration2, _configuration3;
      // Support both direct step types (elasticsearch.search.query) and atomic+configuration pattern
      const stepType = this.step.type || ((_configuration2 = this.step.configuration) === null || _configuration2 === void 0 ? void 0 : _configuration2.type);
      // Use rendered inputs if provided, otherwise fall back to raw step.with or configuration.with
      const stepWith = withInputs || this.step.with || ((_configuration3 = this.step.configuration) === null || _configuration3 === void 0 ? void 0 : _configuration3.with);
      this.workflowLogger.logInfo(`Executing Elasticsearch action: ${stepType}`, {
        event: {
          action: 'elasticsearch-action',
          outcome: 'unknown'
        },
        tags: ['elasticsearch', 'internal-action'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'elasticsearch'
        }
      });

      // Get ES client (user-scoped if available, fallback otherwise)
      const esClient = this.contextManager.getEsClientAsUser();

      // Generic approach like Dev Console - just forward the request to ES
      const result = await this.executeElasticsearchRequest(esClient, stepType, stepWith);
      this.workflowLogger.logInfo(`Elasticsearch action completed: ${stepType}`, {
        event: {
          action: 'elasticsearch-action',
          outcome: 'success'
        },
        tags: ['elasticsearch', 'internal-action'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'elasticsearch'
        }
      });
      return {
        input: stepWith,
        output: result,
        error: undefined
      };
    } catch (error) {
      var _configuration4, _configuration5;
      const stepType = ((_configuration4 = this.step.configuration) === null || _configuration4 === void 0 ? void 0 : _configuration4.type) || this.step.type;
      const stepWith = withInputs || this.step.with || ((_configuration5 = this.step.configuration) === null || _configuration5 === void 0 ? void 0 : _configuration5.with);
      this.workflowLogger.logError(`Elasticsearch action failed: ${stepType}`, error, {
        event: {
          action: 'elasticsearch-action',
          outcome: 'failure'
        },
        tags: ['elasticsearch', 'internal-action', 'error'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'elasticsearch'
        }
      });
      return await this.handleFailure(stepWith, error);
    }
  }
  async executeElasticsearchRequest(esClient, stepType, params) {
    // Support both raw API format and connector-driven syntax
    if (params.request) {
      // Raw API format: { request: { method, path, body } } - like Dev Console
      const {
        method = 'GET',
        path,
        body
      } = params.request;
      return await esClient.transport.request({
        method,
        path,
        body
      });
    } else if (stepType === 'elasticsearch.request') {
      // Special case: elasticsearch.request type uses raw API format at top level
      const {
        method = 'GET',
        path,
        body
      } = params;
      return await esClient.transport.request({
        method,
        path,
        body
      });
    } else {
      // Use generated connector definitions to determine method and path (covers all 568+ ES APIs)
      const {
        method,
        path,
        body: requestBody,
        params: queryParams
      } = (0, _workflows.buildRequestFromConnector)(stepType, params);

      // Build query string manually if needed
      let finalPath = path;
      if (queryParams && Object.keys(queryParams).length > 0) {
        const queryString = new URLSearchParams(queryParams).toString();
        finalPath = `${path}?${queryString}`;
      }
      const requestOptions = {
        method,
        path: finalPath,
        body: requestBody
      };

      // console.log('DEBUG - Sending to ES client:', JSON.stringify(requestOptions, null, 2));
      if (requestOptions.path.endsWith('/_bulk')) {
        var _queryParams$refresh;
        // console.log('DEBUG - Bulk request detected:', JSON.stringify(requestOptions.body, null, 2));
        // Further processing for bulk requests can be added here
        // SG: ugly hack cuz _bulk is special
        const docs = requestOptions.body.operations; // your 3 doc objects
        // If the index is in the path `/tin-workflows/_bulk`, pass it explicitly:
        const pathIndex = requestOptions.path.split('/')[1]; // "tin-workflows"

        // Optional: forward query flags like refresh if you have them
        const refresh = (_queryParams$refresh = queryParams === null || queryParams === void 0 ? void 0 : queryParams.refresh) !== null && _queryParams$refresh !== void 0 ? _queryParams$refresh : false;

        // Turn each doc into an action+doc pair
        const bulkBody = docs.flatMap((doc, i) => {
          // If you have ids, use: { index: { _id: doc._id } }
          return [{
            index: {}
          }, doc];
        });
        const resp = await esClient.bulk({
          index: pathIndex,
          // default index for all actions
          refresh,
          // true | false | 'wait_for'
          body: bulkBody // [ {index:{}}, doc, {index:{}}, doc, ... ]
        });

        // Helpful: surface per-item errors if any
        if (resp.errors) {
          /*
          const itemsWithErrors = resp.items
            .map((it: any, idx: number) => ({
              idx,
              action: Object.keys(it)[0],
              result: it[Object.keys(it)[0]],
            }))
            .filter((x: any) => x.result.error);
          */
          // console.error('Bulk had item errors:', itemsWithErrors);
        }
        return resp;
      }
      return await esClient.transport.request(requestOptions);
    }
  }
}
exports.ElasticsearchActionStepImpl = ElasticsearchActionStepImpl;