"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.evaluateKql = evaluateKql;
var _esQuery = require("@kbn/es-query");
var _node_types = require("@kbn/es-query/src/kuery/node_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function evaluateKql(kql, context) {
  const kqlAst = (0, _esQuery.fromKueryExpression)(kql);
  return evaluateRecursive(kqlAst, context);
}
function evaluateRecursive(node, context) {
  if (node.type === 'function') {
    const functionNode = node;
    switch (functionNode.function) {
      case 'and':
        return functionNode.arguments.every(arg => evaluateRecursive(arg, context));
      case 'or':
        return functionNode.arguments.some(arg => evaluateRecursive(arg, context));
      case 'not':
        return !evaluateRecursive(functionNode.arguments[0], context);
      case 'range':
        return visitRange(functionNode, context);
      case 'is':
        return visitIs(functionNode, context);
      default:
        return false;
    }
  }
  return true;
}
function visitIs(node, context) {
  const [leftLiteral, rightLiteral] = node.arguments;
  const {
    value: contextValue,
    pathExists
  } = readContextPath(leftLiteral.value, context);
  if (!pathExists) {
    return false; // Path does not exist in context
  }
  if (rightLiteral.type === _node_types.KQL_NODE_TYPE_WILDCARD) {
    return true;
  } else if (typeof contextValue === 'string') {
    var _rightLiteral$value;
    return wildcardToRegex((_rightLiteral$value = rightLiteral.value) === null || _rightLiteral$value === void 0 ? void 0 : _rightLiteral$value.toString()).test(contextValue);
  }
  try {
    return contextValue === convertLiteralToValue(rightLiteral, typeof contextValue);
  } catch (error) {
    return false;
  }
}
function visitRange(functionNode, context) {
  const [leftRangeLiteral, operator, rightRangeLiteral] = functionNode.arguments;
  const {
    value: leftRangeValue,
    pathExists
  } = readContextPath(leftRangeLiteral.value, context);
  if (!pathExists) {
    return false; // Path does not exist in context
  }
  let rightRangeValue;
  try {
    rightRangeValue = convertLiteralToValue(rightRangeLiteral, typeof leftRangeValue);
  } catch (error) {
    return false;
  }
  switch (operator) {
    case 'gte':
      return leftRangeValue >= rightRangeValue;
    case 'lte':
      return leftRangeValue <= rightRangeValue;
    case 'gt':
      return leftRangeValue > rightRangeValue;
    case 'lt':
      return leftRangeValue < rightRangeValue;
    default:
      throw new Error(`Unsupported range operator: ${operator}`);
  }
}
function readContextPath(propertyPath, context) {
  const strPropertyPath = String(propertyPath); // sometimes it could be boolean or number
  const propertyPathSegments = strPropertyPath.split('.');
  let result = context;
  for (const segment of propertyPathSegments) {
    if (!(segment in result)) {
      return {
        pathExists: false,
        value: undefined
      }; // Path not found in context
    }
    result = result[segment];
  }
  return {
    pathExists: true,
    value: result
  };
}
function convertLiteralToValue(node, expectedType) {
  switch (expectedType) {
    case 'string':
      return String(node.value);
    case 'number':
      return parseFloat(node.value);
    case 'boolean':
      return String(node.value) === 'true';
    default:
      throw new Error(`Unsupported type: ${expectedType}`);
  }
}
function wildcardToRegex(value) {
  const tokenized = value
  // Temporarily replace escaped wildcards with placeholders
  .replace(/\\\\/g, '__ESCAPED_BACKSLASH__') // handles \\ correctly
  .replace(/\\\*/g, '__LITERAL_AST__').replace(/\\\?/g, '__LITERAL_Q__')

  // Escape regex metacharacters (except wildcards)
  .replace(/([.+^${}()|[\]\\])/g, '\\$1')

  // Convert real wildcards
  .replace(/\*/g, '.*').replace(/\?/g, '.')

  // Restore literal wildcards
  .replace(/__LITERAL_AST__/g, '\\*').replace(/__LITERAL_Q__/g, '\\?').replace(/__ESCAPED_BACKSLASH__/g, '\\\\');
  const wildcardPattern = `^${tokenized}$`;
  return new RegExp(wildcardPattern);
}