"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaActionStepImpl = void 0;
var _workflows = require("@kbn/workflows");
var _node_implementation = require("./node_implementation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Extend BaseStep for kibana-specific properties

class KibanaActionStepImpl extends _node_implementation.BaseAtomicNodeImplementation {
  constructor(step, contextManager, workflowRuntime, workflowLogger) {
    super(step, contextManager, undefined, workflowRuntime);
    this.workflowLogger = workflowLogger;
  }
  getInput() {
    var _configuration;
    // Get current context for templating
    const context = this.contextManager.getContext();
    // Render inputs from 'with' - support both direct step.with and step.configuration.with
    const stepWith = this.step.with || ((_configuration = this.step.configuration) === null || _configuration === void 0 ? void 0 : _configuration.with) || {};
    return this.renderObjectTemplate(stepWith, context);
  }

  /**
   * Recursively render the object template.
   * @param obj - The object to render.
   * @param context - The context to use for rendering.
   * @returns The rendered object.
   */
  renderObjectTemplate(obj, context) {
    if (Array.isArray(obj)) {
      return obj.map(item => this.renderObjectTemplate(item, context));
    }
    if (obj && typeof obj === 'object') {
      return Object.entries(obj).reduce((acc, [key, value]) => {
        acc[key] = this.renderObjectTemplate(value, context);
        return acc;
      }, {});
    }
    if (typeof obj === 'string') {
      return this.templatingEngine.render(obj, context);
    }
    return obj;
  }
  async _run(withInputs) {
    try {
      var _configuration2, _configuration3;
      // Support both direct step types (kibana.createCaseDefaultSpace) and atomic+configuration pattern
      const stepType = this.step.type || ((_configuration2 = this.step.configuration) === null || _configuration2 === void 0 ? void 0 : _configuration2.type);
      // Use rendered inputs if provided, otherwise fall back to raw step.with or configuration.with
      const stepWith = withInputs || this.step.with || ((_configuration3 = this.step.configuration) === null || _configuration3 === void 0 ? void 0 : _configuration3.with);
      this.workflowLogger.logInfo(`Executing Kibana action: ${stepType}`, {
        event: {
          action: 'kibana-action',
          outcome: 'unknown'
        },
        tags: ['kibana', 'internal-action'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'kibana'
        }
      });

      // Get Kibana base URL and authentication
      const kibanaUrl = this.getKibanaUrl();
      const authHeaders = this.getAuthHeaders();

      // Generic approach like Dev Console - just forward the request to Kibana
      const result = await this.executeKibanaRequest(kibanaUrl, authHeaders, stepType, stepWith);
      this.workflowLogger.logInfo(`Kibana action completed: ${stepType}`, {
        event: {
          action: 'kibana-action',
          outcome: 'success'
        },
        tags: ['kibana', 'internal-action'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'kibana'
        }
      });
      return {
        input: stepWith,
        output: result,
        error: undefined
      };
    } catch (error) {
      var _configuration4, _configuration5;
      const stepType = ((_configuration4 = this.step.configuration) === null || _configuration4 === void 0 ? void 0 : _configuration4.type) || this.step.type;
      const stepWith = withInputs || this.step.with || ((_configuration5 = this.step.configuration) === null || _configuration5 === void 0 ? void 0 : _configuration5.with);
      this.workflowLogger.logError(`Kibana action failed: ${stepType}`, error, {
        event: {
          action: 'kibana-action',
          outcome: 'failure'
        },
        tags: ['kibana', 'internal-action', 'error'],
        labels: {
          step_type: stepType,
          connector_type: stepType,
          action_type: 'kibana'
        }
      });
      return await this.handleFailure(stepWith, error);
    }
  }
  getKibanaUrl() {
    var _coreStart$http, _coreStart$http$baseP;
    // Get Kibana URL from CoreStart if available
    const coreStart = this.contextManager.getCoreStart();
    if (coreStart !== null && coreStart !== void 0 && (_coreStart$http = coreStart.http) !== null && _coreStart$http !== void 0 && (_coreStart$http$baseP = _coreStart$http.basePath) !== null && _coreStart$http$baseP !== void 0 && _coreStart$http$baseP.publicBaseUrl) {
      return coreStart.http.basePath.publicBaseUrl;
    }

    // Fallback to localhost for development
    return 'http://localhost:5601';
  }
  getAuthHeaders() {
    var _fakeRequest$headers;
    const headers = {
      'Content-Type': 'application/json',
      'kbn-xsrf': 'true'
    };

    // Get fakeRequest for authentication (created by Task Manager from taskInstance.apiKey)
    const fakeRequest = this.contextManager.getFakeRequest();
    if (fakeRequest !== null && fakeRequest !== void 0 && (_fakeRequest$headers = fakeRequest.headers) !== null && _fakeRequest$headers !== void 0 && _fakeRequest$headers.authorization) {
      // Use API key from fakeRequest if available
      headers.Authorization = fakeRequest.headers.authorization.toString();
    } else {
      // Fallback to basic auth for development
      const basicAuth = Buffer.from('elastic:changeme').toString('base64');
      headers.Authorization = `Basic ${basicAuth}`;
    }

    // Note: User context is not available in KibanaRequestAuth interface
    // Could be added in the future if needed for user attribution

    return headers;
  }
  async executeKibanaRequest(kibanaUrl, authHeaders, stepType, params) {
    // Support both raw API format and connector-driven syntax
    if (params.request) {
      // Raw API format: { request: { method, path, body, query, headers } } - like Dev Console
      const {
        method = 'GET',
        path,
        body,
        query,
        headers: customHeaders
      } = params.request;
      return await this.makeHttpRequest(kibanaUrl, {
        method,
        path,
        body,
        query,
        headers: {
          ...authHeaders,
          ...customHeaders
        }
      });
    } else {
      // Use generated connector definitions to determine method and path (covers all 454+ Kibana APIs)
      const {
        method,
        path,
        body,
        query,
        headers: connectorHeaders
      } = (0, _workflows.buildKibanaRequestFromAction)(stepType, params);
      return await this.makeHttpRequest(kibanaUrl, {
        method,
        path,
        body,
        query,
        headers: {
          ...authHeaders,
          ...connectorHeaders
        }
      });
    }
  }
  async makeHttpRequest(kibanaUrl, requestConfig) {
    const {
      method,
      path,
      body,
      query,
      headers = {}
    } = requestConfig;

    // Build full URL with query parameters
    let fullUrl = `${kibanaUrl}${path}`;
    if (query && Object.keys(query).length > 0) {
      const queryString = new URLSearchParams(query).toString();
      fullUrl = `${fullUrl}?${queryString}`;
    }
    const response = await fetch(fullUrl, {
      method,
      headers,
      body: body ? JSON.stringify(body) : undefined
    });
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`HTTP ${response.status}: ${errorText}`);
    }
    const responseData = await response.json();
    return responseData;
  }
}
exports.KibanaActionStepImpl = KibanaActionStepImpl;