"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WaitStepImpl = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WaitStepImpl {
  constructor(node, stepContext, workflowRuntime, workflowLogger, workflowTaskManager) {
    // 5 seconds
    (0, _defineProperty2.default)(this, "durationCache", null);
    this.node = node;
    this.stepContext = stepContext;
    this.workflowRuntime = workflowRuntime;
    this.workflowLogger = workflowLogger;
    this.workflowTaskManager = workflowTaskManager;
  }
  async run() {
    if (this.getDurationInMs() > WaitStepImpl.SHORT_DURATION_THRESHOLD) {
      await this.handleLongDuration();
      return;
    }
    await this.handleShortDuration();
  }
  getDurationInMs() {
    if (typeof this.durationCache === 'number') {
      return this.durationCache;
    }
    return this.durationCache = (0, _utils.parseDuration)(this.node.configuration.with.duration);
  }
  async handleShortDuration() {
    await this.workflowRuntime.startStep();
    this.logStartWait();
    const durationInMs = this.getDurationInMs();
    await new Promise((resolve, reject) => {
      const timeoutId = setTimeout(resolve, durationInMs);
      this.stepContext.abortController.signal.addEventListener('abort', () => {
        clearTimeout(timeoutId);
        reject(new Error('Wait step was aborted'));
      });
    });
    this.logFinishWait();
    await this.workflowRuntime.finishStep();
    this.workflowRuntime.navigateToNextNode();
  }
  handleLongDuration() {
    const stepState = this.workflowRuntime.getCurrentStepState();
    if (stepState !== null && stepState !== void 0 && stepState.resumeExecutionTaskId) {
      return this.exitLongWait();
    }
    return this.enterLongWait();
  }
  async enterLongWait() {
    const durationInMs = this.getDurationInMs();
    await this.workflowRuntime.startStep();
    const workflowExecution = this.workflowRuntime.getWorkflowExecution();
    const runAt = new Date(new Date().getTime() + durationInMs);
    const resumeExecutionTask = await this.workflowTaskManager.scheduleResumeTask({
      runAt,
      workflowRunId: workflowExecution.id,
      spaceId: workflowExecution.spaceId
    });
    this.workflowLogger.logDebug(`Scheduled resume execution task for wait step "${this.node.id}" with ID ${resumeExecutionTask.taskId}.\nExecution will resume at ${runAt.toISOString()}`);
    await this.workflowRuntime.setCurrentStepState({
      resumeExecutionTaskId: resumeExecutionTask.taskId
    });
    this.logStartWait();
    await this.workflowRuntime.setWaitStep();
  }
  async exitLongWait() {
    await this.workflowRuntime.setCurrentStepState(undefined);
    this.workflowLogger.logDebug(`Resuming execution of wait step "${this.node.id}" after long wait.`);
    this.logFinishWait();
    await this.workflowRuntime.finishStep();
    this.workflowRuntime.navigateToNextNode();
  }
  logStartWait() {
    this.workflowLogger.logInfo(`Waiting for ${this.node.configuration.with.duration} in step ${this.node.id}`);
  }
  logFinishWait() {
    this.workflowLogger.logInfo(`Finished waiting for ${this.node.configuration.with.duration} in step ${this.node.id}`);
  }
}
exports.WaitStepImpl = WaitStepImpl;
(0, _defineProperty2.default)(WaitStepImpl, "SHORT_DURATION_THRESHOLD", 1000 * 5);