"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowExecutionState = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowExecutionState {
  constructor(initialWorkflowExecution, workflowExecutionRepository, workflowStepExecutionRepository) {
    (0, _defineProperty2.default)(this, "stepExecutions", new Map());
    (0, _defineProperty2.default)(this, "workflowExecution", void 0);
    (0, _defineProperty2.default)(this, "workflowChanges", []);
    (0, _defineProperty2.default)(this, "stepChanges", []);
    /**
     * Maps step IDs to their execution IDs in chronological order.
     * This index enables efficient lookup of all executions for a given step,
     * which is especially important for steps that execute multiple times
     * (e.g., in loops or retries).
     */
    (0, _defineProperty2.default)(this, "stepIdExecutionIdIndex", new Map());
    this.workflowExecutionRepository = workflowExecutionRepository;
    this.workflowStepExecutionRepository = workflowStepExecutionRepository;
    this.workflowExecution = initialWorkflowExecution;
  }
  async load() {
    const foundSteps = await this.workflowStepExecutionRepository.searchStepExecutionsByExecutionId(this.workflowExecution.id);
    foundSteps.forEach(stepExecution => this.stepExecutions.set(stepExecution.id, stepExecution));
    this.buildStepIdExecutionIdIndex();
  }
  getWorkflowExecution() {
    return this.workflowExecution;
  }
  updateWorkflowExecution(workflowExecution) {
    this.workflowExecution = {
      ...this.workflowExecution,
      ...workflowExecution
    };
    this.workflowChanges.push({
      objectId: this.workflowExecution.id,
      changeType: 'update',
      change: workflowExecution
    });
  }
  getAllStepExecutions() {
    return Array.from(this.stepExecutions.values());
  }
  getStepExecution(stepExecutionId) {
    return this.stepExecutions.get(stepExecutionId);
  }

  /**
   * Retrieves all executions for a specific workflow step in chronological order.
   * @param stepId The unique identifier of the step
   * @returns An array of step execution objects or undefined if no executions exist
   */
  getStepExecutionsByStepId(stepId) {
    if (!this.stepIdExecutionIdIndex.has(stepId)) {
      return [];
    }
    return this.stepIdExecutionIdIndex.get(stepId).map(executionId => this.stepExecutions.get(executionId));
  }

  /**
   * Retrieves the latest execution for a specific workflow step.
   * @param stepId The unique identifier of the step
   * @returns The latest step execution object or undefined if no executions exist
   */
  getLatestStepExecution(stepId) {
    const allExecutions = this.getStepExecutionsByStepId(stepId);
    if (!(allExecutions !== null && allExecutions !== void 0 && allExecutions.length)) {
      return undefined;
    }
    return allExecutions[allExecutions.length - 1];
  }
  upsertStep(step) {
    if (!step.id) {
      throw new Error('WorkflowExecutionState: Step execution must have an ID to be upserted');
    }
    if (!this.stepExecutions.has(step.id)) {
      this.createStep(step);
    } else {
      this.updateStep(step);
    }
  }
  async flushStepChanges() {
    const stepChanges = Array.from(this.stepChanges.values());
    const tasks = [];
    if (stepChanges.length > 0) {
      // const groupedStepChangesById =
      // Group step changes by objectId to handle multiple changes to the same step
      const groupedStepChangesById = stepChanges.reduce((acc, change) => {
        // For each objectId, keep the latest change
        if (!acc[change.objectId]) {
          acc[change.objectId] = [];
        }
        acc[change.objectId].push(change);
        return acc;
      }, {});
      const createSteps = [];
      const updateSteps = [];
      Object.entries(groupedStepChangesById).forEach(([objectId, changes]) => {
        const accumulated = changes.reduce((acc, change) => ({
          ...acc,
          ...change.change
        }), {});
        if (changes.some(change => change.changeType === 'create')) {
          createSteps.push(accumulated);
        } else {
          updateSteps.push(accumulated);
        }
      });
      if (createSteps.length > 0) {
        createSteps.forEach(step => tasks.push(this.workflowStepExecutionRepository.createStepExecution(step)));
      }
      if (updateSteps.length > 0) {
        tasks.push(this.workflowStepExecutionRepository.updateStepExecutions(updateSteps));
      }
    }
    await Promise.all(tasks);
    this.stepChanges = [];
  }
  async flush() {
    await Promise.all([this.flushWorkflowChanges(), this.flushStepChanges()]);
  }
  async flushWorkflowChanges() {
    if (!this.workflowChanges.length) {
      return;
    }
    const accumulated = this.workflowChanges.reduce((prev, acc) => ({
      ...prev,
      ...acc.change
    }), {
      id: this.workflowExecution.id
    });
    await this.workflowExecutionRepository.updateWorkflowExecution(accumulated);
    const fetchedWorkflowExecution = await this.workflowExecutionRepository.getWorkflowExecutionById(this.workflowExecution.id, this.workflowExecution.spaceId);
    this.workflowExecution = fetchedWorkflowExecution;
    this.workflowChanges = [];
  }
  createStep(step) {
    const stepExecutions = this.getStepExecutionsByStepId(step.stepId) || [];
    if (!stepExecutions.length) {
      this.stepIdExecutionIdIndex.set(step.stepId, []);
    }
    this.stepIdExecutionIdIndex.get(step.stepId).push(step.id);
    const newStep = {
      ...step,
      id: step.id,
      globalExecutionIndex: this.stepExecutions.size,
      stepExecutionIndex: stepExecutions.length,
      workflowRunId: this.workflowExecution.id,
      workflowId: this.workflowExecution.workflowId,
      spaceId: this.workflowExecution.spaceId
    };
    this.stepExecutions.set(step.id, newStep);
    this.stepChanges.push({
      objectId: newStep.id,
      changeType: 'create',
      change: newStep
    });
  }
  updateStep(step) {
    this.stepExecutions.set(step.id, {
      ...this.stepExecutions.get(step.id),
      ...step
    });
    this.stepChanges.push({
      objectId: step.id,
      changeType: 'update',
      change: step
    });
  }
  buildStepIdExecutionIdIndex() {
    this.stepIdExecutionIdIndex.clear();
    for (const step of this.stepExecutions.values()) {
      if (!this.stepIdExecutionIdIndex.has(step.stepId)) {
        this.stepIdExecutionIdIndex.set(step.stepId, []);
      }
      this.stepIdExecutionIdIndex.get(step.stepId).push(step.id);
    }
    for (const [stepId, stepExecutionIds] of this.stepIdExecutionIdIndex.entries()) {
      this.stepIdExecutionIdIndex.set(stepId, stepExecutionIds.sort((a, b) => {
        var _aExecution$stepExecu, _bExecution$stepExecu;
        const aExecution = this.stepExecutions.get(a);
        const bExecution = this.stepExecutions.get(b);
        return ((_aExecution$stepExecu = aExecution === null || aExecution === void 0 ? void 0 : aExecution.stepExecutionIndex) !== null && _aExecution$stepExecu !== void 0 ? _aExecution$stepExecu : 0) - ((_bExecution$stepExecu = bExecution === null || bExecution === void 0 ? void 0 : bExecution.stepExecutionIndex) !== null && _bExecution$stepExecu !== void 0 ? _bExecution$stepExecu : 0);
      }));
    }
  }
}
exports.WorkflowExecutionState = WorkflowExecutionState;