"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowEventLogger = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowEventLogger {
  constructor(esClient, logger, indexName, context = {}, options = {}) {
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "indexName", void 0);
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "options", void 0);
    (0, _defineProperty2.default)(this, "eventQueue", []);
    (0, _defineProperty2.default)(this, "timings", new Map());
    this.esClient = esClient;
    this.logger = logger;
    this.indexName = indexName;
    this.context = context;
    this.options = options;
  }
  logEvent(eventProperties) {
    const event = this.createBaseEvent();

    // Merge context, default properties, and provided properties
    (0, _lodash.merge)(event, eventProperties);

    // Log to console if enabled
    if (this.options.enableConsoleLogging) {
      this.logToConsole(event);
    }
    const doc = {
      index: this.indexName,
      body: event
    };
    this.queueEvent(doc);
  }
  logInfo(message, additionalData = {}) {
    this.logEvent({
      message,
      level: 'info',
      event: {
        action: 'log',
        category: ['workflow'],
        type: ['info'],
        provider: 'workflow-engine',
        ...additionalData.event
      },
      ...additionalData
    });
  }
  logError(message, error, additionalData = {}) {
    const errorData = {};
    if (error) {
      errorData.error = {
        message: error.message,
        type: error.name,
        stack_trace: error.stack
      };
    }
    this.logEvent({
      message,
      level: 'error',
      event: {
        action: 'error',
        category: ['workflow'],
        type: ['error'],
        provider: 'workflow-engine',
        outcome: 'failure',
        ...additionalData.event
      },
      ...errorData,
      ...additionalData
    });
  }
  logWarn(message, additionalData = {}) {
    this.logEvent({
      message,
      level: 'warn',
      event: {
        action: 'warning',
        category: ['workflow'],
        type: ['info'],
        provider: 'workflow-engine',
        ...additionalData.event
      },
      ...additionalData
    });
  }
  logDebug(message, additionalData = {}) {
    this.logEvent({
      message,
      level: 'debug',
      event: {
        action: 'debug',
        category: ['workflow'],
        type: ['info'],
        provider: 'workflow-engine',
        ...additionalData.event
      },
      ...additionalData
    });
  }
  startTiming(event) {
    var _event$event;
    const timingKey = this.getTimingKey(event);
    this.timings.set(timingKey, new Date());

    // Log start event
    this.logEvent({
      ...event,
      event: {
        ...event.event,
        action: (_event$event = event.event) !== null && _event$event !== void 0 && _event$event.action ? `${event.event.action}-start` : 'timing-start',
        start: new Date().toISOString()
      }
    });
  }
  stopTiming(event) {
    const timingKey = this.getTimingKey(event);
    const startTime = this.timings.get(timingKey);
    if (startTime) {
      var _event$event2, _event$event3;
      const endTime = new Date();
      const duration = endTime.getTime() - startTime.getTime();
      this.logEvent({
        ...event,
        event: {
          ...event.event,
          action: (_event$event2 = event.event) !== null && _event$event2 !== void 0 && _event$event2.action ? `${event.event.action}-complete` : 'timing-complete',
          start: startTime.toISOString(),
          end: endTime.toISOString(),
          duration,
          outcome: ((_event$event3 = event.event) === null || _event$event3 === void 0 ? void 0 : _event$event3.outcome) || 'success'
        }
      });
      this.timings.delete(timingKey);
    }
  }
  createStepLogger(stepExecutionId, stepId, stepName, stepType) {
    return new WorkflowEventLogger(this.esClient, this.logger, this.indexName, {
      ...this.context,
      stepExecutionId,
      stepId,
      stepName,
      stepType
    }, this.options);
  }
  createBaseEvent() {
    return {
      '@timestamp': new Date().toISOString(),
      spaceId: this.context.spaceId,
      workflow: {
        id: this.context.workflowId,
        name: this.context.workflowName,
        execution_id: this.context.executionId,
        step_id: this.context.stepId,
        step_execution_id: this.context.stepExecutionId,
        step_name: this.context.stepName,
        step_type: this.context.stepType
      },
      event: {
        provider: 'workflow-engine'
      },
      tags: ['workflow']
    };
  }
  logToConsole(event) {
    var _event$workflow, _event$workflow2, _event$workflow3, _event$workflow4;
    const message = event.message || '';

    // Format workflow context metadata
    const meta = {
      workflow: {
        name: (_event$workflow = event.workflow) === null || _event$workflow === void 0 ? void 0 : _event$workflow.name,
        execution_id: (_event$workflow2 = event.workflow) === null || _event$workflow2 === void 0 ? void 0 : _event$workflow2.execution_id,
        step_name: (_event$workflow3 = event.workflow) === null || _event$workflow3 === void 0 ? void 0 : _event$workflow3.step_name,
        step_type: (_event$workflow4 = event.workflow) === null || _event$workflow4 === void 0 ? void 0 : _event$workflow4.step_type
      },
      event: event.event,
      tags: event.tags
    };

    // Add error details if present
    if (event.error) {
      meta.error = event.error;
    }

    // Use Kibana's structured logger with appropriate level
    switch (event.level) {
      case 'error':
        this.logger.error(`🔄 WORKFLOW: ${message}`, meta);
        break;
      case 'warn':
        this.logger.warn(`🔄 WORKFLOW: ${message}`, meta);
        break;
      case 'debug':
        this.logger.debug(`🔄 WORKFLOW: ${message}`, meta);
        break;
      case 'trace':
        this.logger.trace(`🔄 WORKFLOW: ${message}`, meta);
        break;
      default:
        this.logger.info(`🔄 WORKFLOW: ${message}`, meta);
    }
  }
  getTimingKey(event) {
    var _event$event4;
    return `${this.context.executionId || 'unknown'}-${((_event$event4 = event.event) === null || _event$event4 === void 0 ? void 0 : _event$event4.action) || 'unknown'}-${this.context.stepId || 'workflow'}`;
  }
  queueEvent(doc) {
    this.eventQueue.push(doc);
  }
  async flushEvents() {
    if (this.eventQueue.length === 0) return;
    const events = [...this.eventQueue];
    this.eventQueue = [];
    try {
      const bulkBody = [];
      for (const doc of events) {
        bulkBody.push({
          create: {}
        });
        bulkBody.push(doc.body);
      }
      await this.esClient.bulk({
        index: this.indexName,
        body: bulkBody,
        refresh: 'wait_for'
      });
      this.logger.info(`Successfully indexed ${events.length} workflow events`);
    } catch (error) {
      this.logger.error(`Failed to index workflow events: ${error.message}`, {
        eventsCount: events.length,
        error: error.stack
      });

      // Re-queue events for retry (optional)
      this.eventQueue.unshift(...events);
    }
  }
}
exports.WorkflowEventLogger = WorkflowEventLogger;