"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cancelWorkflowIfRequested = cancelWorkflowIfRequested;
var _workflows = require("@kbn/workflows");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * This function retrieves the current workflow execution and verifies if cancellation requested.
 * In case when cancelRequested is true, it aborts the monitoredContext.abortController and marks the workflow as cancelled.
 * When monitoredContext.abortController.abort() is called, it will send cancellation signal to currently running node/step,
 * and in case if the node/step supports cancellation (like HTTP step with AbortSignal), it will stop its execution immediately.
 */
async function cancelWorkflowIfRequested(workflowExecutionRepository, workflowExecutionState, monitoredContext, monitorAbortController) {
  if (!workflowExecutionState.getWorkflowExecution().cancelRequested) {
    const currentExecution = await workflowExecutionRepository.getWorkflowExecutionById(workflowExecutionState.getWorkflowExecution().id, workflowExecutionState.getWorkflowExecution().spaceId);
    if (!(currentExecution !== null && currentExecution !== void 0 && currentExecution.cancelRequested)) {
      return;
    }
  }
  monitorAbortController === null || monitorAbortController === void 0 ? void 0 : monitorAbortController.abort();
  monitoredContext.abortController.abort();
  let nodeStack = monitoredContext.scopeStack;

  // mark current step scopes as cancelled
  while (!nodeStack.isEmpty()) {
    const scopeData = nodeStack.getCurrentScope();
    nodeStack = nodeStack.exitScope();
    const stepExecutionId = (0, _utils.buildStepExecutionId)(workflowExecutionState.getWorkflowExecution().id, scopeData.stepId, nodeStack.stackFrames);
    if (workflowExecutionState.getStepExecution(stepExecutionId)) {
      workflowExecutionState.upsertStep({
        id: stepExecutionId,
        status: _workflows.ExecutionStatus.CANCELLED
      });
    }
  }
  workflowExecutionState.upsertStep({
    id: monitoredContext.stepExecutionId,
    status: _workflows.ExecutionStatus.CANCELLED
  });
  workflowExecutionState.updateWorkflowExecution({
    status: _workflows.ExecutionStatus.CANCELLED
  });
}