"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runNode = runNode;
var _workflow_context_manager = require("../workflow_context_manager/workflow_context_manager");
var _run_stack_monitor = require("./run_stack_monitor");
var _catch_error = require("./catch_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Executes a single step in the workflow execution process.
 *
 * This function orchestrates the execution of a workflow node by:
 * 1. Creating a context manager for the current step
 * 2. Creating and running the node implementation
 * 3. Running monitoring in parallel to handle cancellation, timeouts, and other control flow
 * 4. Managing error handling and state persistence
 *
 * The execution uses a race condition between the step execution and monitoring to ensure
 * proper cancellation and timeout handling. Monitoring can prevent step execution if
 * conditions like workflow cancellation or timeout occur before the step runs.
 *
 * @param params - The workflow execution loop parameters containing:
 *   - workflowRuntime: Runtime instance managing workflow state and navigation
 *   - workflowExecutionGraph: The workflow graph definition
 *   - workflowExecutionState: Current execution state
 *   - nodesFactory: Factory for creating node implementations
 *   - esClient: Elasticsearch client for data operations
 *   - fakeRequest: Request object for context
 *   - coreStart: Kibana core services
 *
 * @returns Promise that resolves when the step execution is complete
 *
 * @throws Will catch and handle errors through the workflow runtime's error handling mechanism
 */
async function runNode(params) {
  const currentNode = params.workflowRuntime.getCurrentNode();
  const stepContext = new _workflow_context_manager.WorkflowContextManager({
    workflowExecutionGraph: params.workflowExecutionGraph,
    workflowExecutionState: params.workflowExecutionState,
    esClient: params.esClient,
    fakeRequest: params.fakeRequest,
    coreStart: params.coreStart,
    node: currentNode,
    stackFrames: params.workflowRuntime.getCurrentNodeScope()
  });
  const nodeImplementation = params.nodesFactory.create(stepContext);
  const monitorAbortController = new AbortController();

  // The order of these promises is important - we want to stop monitoring
  const runMonitorPromise = (0, _run_stack_monitor.runStackMonitor)(params, stepContext, monitorAbortController);
  let runStepPromise = Promise.resolve();

  // Sometimes monitoring can prevent the step from running, e.g. when the workflow is cancelled, timeout occured right before running step, etc.
  if (!monitorAbortController.signal.aborted) {
    runStepPromise = nodeImplementation.run().then(() => monitorAbortController.abort());
  }
  try {
    await Promise.race([runMonitorPromise, runStepPromise]);
    monitorAbortController.abort();
  } catch (error) {
    params.workflowRuntime.setWorkflowError(error);
  } finally {
    monitorAbortController.abort();
    await (0, _catch_error.catchError)(params, stepContext);
    await params.workflowRuntime.saveState(); // Ensure state is updated after each step
  }
}