"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectZodSchemaEqual = expectZodSchemaEqual;
exports.getSchemaAtPath = getSchemaAtPath;
exports.getZodTypeName = getZodTypeName;
exports.inferZodType = inferZodType;
exports.isValidSchemaPath = isValidSchemaPath;
exports.parsePath = parsePath;
var _zod = require("@kbn/zod");
var _zodToJsonSchema = require("zod-to-json-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function parsePath(path) {
  const segments = path.replace(/\[(['"]?)([^\]]+)\1\]/g, '.$2') // Convert [key] to .key
  .split('.');
  return segments.some(s => s === '') ? null : segments;
}

/**
 * Get zod schema at a given path.
 * @param schema - The zod schema to get the path from.
 * @param path - The path to get the schema from. e.g. `choices[0].message['content']`
 * @param options - The options for the function.
 * @param options.partial - If true, return the schema for the last valid path segment.
 * @returns The schema at the given path or null if the path is invalid.
 */
function getSchemaAtPath(schema, path, {
  partial = false
} = {}) {
  try {
    const segments = parsePath(path);
    if (!segments) {
      return null;
    }
    let current = schema;
    for (const segment of segments) {
      if (current instanceof _zod.z.ZodOptional) {
        current = current.unwrap();
      }
      if (current instanceof _zod.z.ZodObject) {
        const shape = current.shape;
        if (!(segment in shape)) {
          return partial ? current : null;
        }
        current = shape[segment];
      } else if (current instanceof _zod.z.ZodUnion) {
        const branches = current.options;
        const validBranch = branches.find(branch => isValidSchemaPath(branch, segment));
        if (!validBranch) {
          return partial ? current : null;
        }
        current = validBranch;
      } else if (current instanceof _zod.z.ZodArray) {
        var _current$_def$maxLeng, _current$_def$maxLeng2, _current$_def$exactLe;
        if (!/^\d+$/.test(segment)) {
          return partial ? current : null;
        }
        const index = parseInt(segment, 10);

        // Reject negative indices
        if (index < 0) {
          return partial ? current : null;
        }

        // Only enforce bounds checking for arrays with explicit length constraints
        const maxLength = (_current$_def$maxLeng = (_current$_def$maxLeng2 = current._def.maxLength) === null || _current$_def$maxLeng2 === void 0 ? void 0 : _current$_def$maxLeng2.value) !== null && _current$_def$maxLeng !== void 0 ? _current$_def$maxLeng : (_current$_def$exactLe = current._def.exactLength) === null || _current$_def$exactLe === void 0 ? void 0 : _current$_def$exactLe.value;
        if (maxLength !== undefined && index >= maxLength) {
          return partial ? current : null;
        }

        // For unconstrained arrays, we allow any non-negative index for schema introspection
        // This is because we're validating schema paths, not runtime data
        current = current.element;
      } else if (current instanceof _zod.z.ZodAny) {
        return _zod.z.any();
      } else if (current instanceof _zod.z.ZodUnknown) {
        return _zod.z.unknown();
      } else {
        return null;
      }
    }
    if (current instanceof _zod.z.ZodOptional) {
      return current.unwrap();
    }
    return current;
  } catch {
    return null;
  }
}

/**
 * Check if a path is valid for a given zod schema.
 * @param schema - The zod schema to check the path for.
 * @param path - The path to check. e.g. `choices[0].message['content']`
 * @returns True if the path is valid, false otherwise.
 */
function isValidSchemaPath(schema, path) {
  return getSchemaAtPath(schema, path) !== null;
}

/**
 * Infer a zod schema from an object.
 * @param obj - The object to infer the schema from.
 * @returns The inferred zod schema.
 */
function inferZodType(obj) {
  if (obj === null) return _zod.z.null();
  if (obj === undefined) return _zod.z.undefined();
  const type = typeof obj;
  if (type === 'string') return _zod.z.string();
  if (type === 'number') return _zod.z.number();
  if (type === 'boolean') return _zod.z.boolean();
  if (Array.isArray(obj)) {
    if (obj.length === 0) return _zod.z.array(_zod.z.unknown());
    return _zod.z.array(inferZodType(obj[0])).length(obj.length);
  }
  if (type === 'object') {
    const shape = {};
    for (const [key, value] of Object.entries(obj)) {
      shape[key] = inferZodType(value);
    }
    return _zod.z.object(shape);
  }
  return _zod.z.unknown();
}
function expectZodSchemaEqual(a, b) {
  expect((0, _zodToJsonSchema.zodToJsonSchema)(a)).toEqual((0, _zodToJsonSchema.zodToJsonSchema)(b));
}
function getZodTypeName(schema) {
  const typedSchema = schema;
  const def = typedSchema._def;
  switch (def.typeName) {
    case 'ZodString':
      return 'string';
    case 'ZodNumber':
      return 'number';
    case 'ZodBoolean':
      return 'boolean';
    case 'ZodArray':
      return 'array';
    case 'ZodObject':
      return 'object';
    case 'ZodDate':
      return 'date';
    case 'ZodAny':
      return 'any';
    case 'ZodNull':
      return 'null';
    case 'ZodUnknown':
      return 'unknown';
    default:
      return 'unknown';
  }
}