"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKFLOW_ZOD_SCHEMA_LOOSE = exports.WORKFLOW_ZOD_SCHEMA = void 0;
exports.getAllConnectors = getAllConnectors;
exports.getWorkflowZodSchemaLoose = exports.getWorkflowZodSchema = exports.getOutputSchemaForStepType = void 0;
var _i18n = require("@kbn/i18n");
var _workflows = require("@kbn/workflows");
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Static connectors used for schema generation
const staticConnectors = [{
  type: 'console',
  paramsSchema: _zod.z.object({
    message: _zod.z.string()
  }).required(),
  outputSchema: _zod.z.string(),
  description: _i18n.i18n.translate('workflows.connectors.console.description', {
    defaultMessage: 'Log a message to the workflow logs'
  })
}, {
  type: 'slack',
  connectorIdRequired: true,
  paramsSchema: _zod.z.object({
    message: _zod.z.string()
  }).required(),
  outputSchema: _zod.z.object({
    message: _zod.z.string()
  }),
  description: _i18n.i18n.translate('workflows.connectors.slack.description', {
    defaultMessage: 'Send a message to a Slack channel'
  })
}, {
  type: 'inference.unified_completion',
  connectorIdRequired: true,
  paramsSchema: _zod.z.object({
    body: _zod.z.object({
      messages: _zod.z.array(_zod.z.object({
        role: _zod.z.string(),
        content: _zod.z.string()
      }))
    })
  }).required(),
  // TODO: use UnifiedChatCompleteResponseSchema from stack_connectors/common/inference/schema.ts
  outputSchema: _zod.z.object({
    id: _zod.z.string(),
    choices: _zod.z.array(_zod.z.object({
      message: _zod.z.object({
        content: _zod.z.string(),
        role: _zod.z.string()
      })
    }))
  }),
  description: _i18n.i18n.translate('workflows.connectors.inference.unified_completion.description', {
    defaultMessage: 'Generate text using a model with advanced input parameters'
  })
}, {
  type: 'inference.completion',
  connectorIdRequired: true,
  paramsSchema: _zod.z.object({
    input: _zod.z.string()
  }),
  outputSchema: _zod.z.array(_zod.z.object({
    result: _zod.z.string()
  })),
  description: _i18n.i18n.translate('workflows.connectors.inference.completion.description', {
    defaultMessage: 'Generate text using a model'
  })
},
// Generic request types for raw API calls
{
  type: 'elasticsearch.request',
  connectorIdRequired: false,
  paramsSchema: _zod.z.object({
    method: _zod.z.string(),
    path: _zod.z.string(),
    body: _zod.z.any().optional(),
    params: _zod.z.any().optional()
  }),
  outputSchema: _zod.z.any(),
  description: _i18n.i18n.translate('workflows.connectors.elasticsearch.request.description', {
    defaultMessage: 'Make a generic request to an Elasticsearch API'
  })
}, {
  type: 'kibana.request',
  connectorIdRequired: false,
  paramsSchema: _zod.z.object({
    method: _zod.z.string(),
    path: _zod.z.string(),
    body: _zod.z.any().optional(),
    headers: _zod.z.any().optional()
  }),
  outputSchema: _zod.z.any(),
  description: _i18n.i18n.translate('workflows.connectors.kibana.request.description', {
    defaultMessage: 'Make a generic request to a Kibana API'
  })
}];

/**
 * Elasticsearch Connector Generation
 *
 * This system provides ConnectorContract[] for all Elasticsearch APIs
 * by using pre-generated definitions from Console's API specifications.
 *
 * Benefits:
 * 1. **Schema Validation**: Zod schemas for all ES API parameters
 * 2. **Autocomplete**: Monaco YAML editor gets full autocomplete via JSON Schema
 * 3. **Comprehensive Coverage**: 568 Elasticsearch APIs supported
 * 4. **Browser Compatible**: No file system access required
 * 5. **Lazy Loading**: Large generated files are only loaded when needed, reducing main bundle size
 *
 * The generated connectors include:
 * - All Console API definitions converted to Zod schemas
 * - Path parameters extracted from patterns (e.g., {index}, {id})
 * - URL parameters with proper types (flags, enums, strings, numbers)
 * - Proper ConnectorContract format for workflow execution
 *
 * To regenerate: run `npm run generate:es-connectors` from @kbn/workflows package
 */
function generateElasticsearchConnectors() {
  // Lazy load the large generated files to keep them out of the main bundle
  const {
    GENERATED_ELASTICSEARCH_CONNECTORS
    // eslint-disable-next-line @typescript-eslint/no-var-requires
  } = require('@kbn/workflows/common/generated_es_connectors');
  const {
    ENHANCED_ELASTICSEARCH_CONNECTORS,
    mergeEnhancedConnectors
    // eslint-disable-next-line @typescript-eslint/no-var-requires
  } = require('./enhanced_es_connectors');

  // Return enhanced connectors (merge generated with enhanced definitions)
  return mergeEnhancedConnectors(GENERATED_ELASTICSEARCH_CONNECTORS, ENHANCED_ELASTICSEARCH_CONNECTORS);
}
function generateKibanaConnectors() {
  // Lazy load the generated Kibana connectors

  const {
    GENERATED_KIBANA_CONNECTORS
    // eslint-disable-next-line @typescript-eslint/no-var-requires
  } = require('@kbn/workflows/common/generated_kibana_connectors');

  // Return the pre-generated Kibana connectors (build-time generated, browser-safe)
  return GENERATED_KIBANA_CONNECTORS;
}

// Combine static connectors with dynamic Elasticsearch and Kibana connectors
function getAllConnectors() {
  const elasticsearchConnectors = generateElasticsearchConnectors();
  const kibanaConnectors = generateKibanaConnectors();
  return [...staticConnectors, ...elasticsearchConnectors, ...kibanaConnectors];
}
const getOutputSchemaForStepType = stepType => {
  const allConnectors = getAllConnectors();
  const connector = allConnectors.find(c => c.type === stepType);
  if (connector) {
    return connector.outputSchema;
  }

  // Handle internal actions with pattern matching
  // TODO: add output schema support for elasticsearch.request and kibana.request connectors
  if (stepType.startsWith('elasticsearch.')) {
    return _zod.z.any();
  }
  if (stepType.startsWith('kibana.')) {
    return _zod.z.any();
  }

  // Fallback to any if not found
  return _zod.z.any();
};

// Dynamic schemas that include all connectors (static + Elasticsearch)
// These use lazy loading to keep large generated files out of the main bundle
exports.getOutputSchemaForStepType = getOutputSchemaForStepType;
const getWorkflowZodSchema = () => {
  const allConnectors = getAllConnectors();
  return (0, _workflows.generateYamlSchemaFromConnectors)(allConnectors);
};
exports.getWorkflowZodSchema = getWorkflowZodSchema;
const getWorkflowZodSchemaLoose = () => {
  const allConnectors = getAllConnectors();
  return (0, _workflows.generateYamlSchemaFromConnectors)(allConnectors, true);
};

// Legacy exports for backward compatibility - these will be deprecated
// TODO: Remove these once all consumers are updated to use the lazy-loaded versions
exports.getWorkflowZodSchemaLoose = getWorkflowZodSchemaLoose;
const WORKFLOW_ZOD_SCHEMA = exports.WORKFLOW_ZOD_SCHEMA = (0, _workflows.generateYamlSchemaFromConnectors)(staticConnectors);
const WORKFLOW_ZOD_SCHEMA_LOOSE = exports.WORKFLOW_ZOD_SCHEMA_LOOSE = (0, _workflows.generateYamlSchemaFromConnectors)(staticConnectors, true);

// Partially recreated from x-pack/platform/plugins/shared/alerting/server/connector_adapters/types.ts
// TODO: replace with dynamic schema