"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkflowActions = useWorkflowActions;
var _reactQuery = require("@kbn/react-query");
var _public = require("@kbn/kibana-react-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function useWorkflowActions() {
  const queryClient = (0, _reactQuery.useQueryClient)();
  const {
    http
  } = (0, _public.useKibana)().services;
  const createWorkflow = (0, _reactQuery.useMutation)({
    networkMode: 'always',
    mutationKey: ['POST', 'workflows'],
    mutationFn: workflow => {
      return http.post('/api/workflows', {
        body: JSON.stringify(workflow)
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['workflows']
      });
    }
  });
  const updateWorkflow = (0, _reactQuery.useMutation)({
    mutationKey: ['PUT', 'workflows', 'id'],
    mutationFn: ({
      id,
      workflow
    }) => {
      return http.put(`/api/workflows/${id}`, {
        body: JSON.stringify(workflow)
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['workflows']
      });
    }
  });
  const deleteWorkflows = (0, _reactQuery.useMutation)({
    mutationKey: ['DELETE', 'workflows'],
    mutationFn: ({
      ids
    }) => {
      return http.delete(`/api/workflows`, {
        body: JSON.stringify({
          ids
        })
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['workflows']
      });
    }
  });
  const runWorkflow = (0, _reactQuery.useMutation)({
    mutationKey: ['POST', 'workflows', 'id', 'run'],
    mutationFn: ({
      id,
      inputs
    }) => {
      return http.post(`/api/workflows/${id}/run`, {
        body: JSON.stringify({
          inputs
        })
      });
    },
    onSuccess: (_, {
      id
    }) => {
      // FIX: ensure workflow execution document is created at the end of the mutation
      queryClient.invalidateQueries({
        queryKey: ['workflows']
      });
      queryClient.invalidateQueries({
        queryKey: ['workflows', id, 'executions']
      });
      queryClient.invalidateQueries({
        queryKey: ['workflows', id]
      });
    }
  });
  const runIndividualStep = (0, _reactQuery.useMutation)({
    mutationKey: ['POST', 'workflows', 'stepId', 'run'],
    mutationFn: ({
      stepId,
      contextOverride,
      workflowYaml
    }) => {
      return http.post(`/api/workflows/testStep`, {
        body: JSON.stringify({
          stepId,
          contextOverride,
          workflowYaml
        })
      });
    },
    onSuccess: ({
      workflowExecutionId
    }, {}) => {
      queryClient.invalidateQueries({
        queryKey: ['workflows', workflowExecutionId, 'executions']
      });
    }
  });
  const cloneWorkflow = (0, _reactQuery.useMutation)({
    mutationKey: ['POST', 'workflows', 'id', 'clone'],
    mutationFn: ({
      id
    }) => {
      return http.post(`/api/workflows/${id}/clone`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({
        queryKey: ['workflows']
      });
    }
  });
  const testWorkflow = (0, _reactQuery.useMutation)({
    mutationKey: ['POST', 'workflows', 'test'],
    mutationFn: ({
      workflowYaml,
      inputs
    }) => {
      return http.post(`/api/workflows/test`, {
        body: JSON.stringify({
          workflowYaml,
          inputs
        })
      });
    }
  });
  return {
    createWorkflow,
    updateWorkflow,
    // kc: maybe return mutation.mutate? where the navigation is handled?
    deleteWorkflows,
    runWorkflow,
    runIndividualStep,
    cloneWorkflow,
    testWorkflow
  };
}