"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertWorkflowGraphToReactFlow = convertWorkflowGraphToReactFlow;
var _dagre = _interopRequireDefault(require("@dagrejs/dagre"));
var _react = require("@xyflow/react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Converts a workflow graph into positioned ReactFlow nodes and edges using Dagre layout algorithm.
 *
 * @param graph - The dagre graph representation of the workflow
 * @returns Object containing positioned nodes and edges for ReactFlow
 */
function convertWorkflowGraphToReactFlow(graph) {
  const topologySort = _dagre.default.graphlib.alg.topsort(graph);
  const dagreGraph = new _dagre.default.graphlib.Graph();
  dagreGraph.setGraph({});
  dagreGraph.setDefaultEdgeLabel(() => ({}));

  // Set graph direction and spacing
  dagreGraph.setGraph({
    rankdir: 'BT',
    // Bottom-to-Top direction (reversed)
    nodesep: 40,
    ranksep: 40,
    edgesep: 40,
    marginx: 20,
    marginy: 20
  });
  const stack = [];
  const baseWidth = 100;
  let maxDepth = 0;
  topologySort.map(nodeId => graph.node(nodeId)).forEach(node => {
    if (node.type.startsWith('exit')) {
      stack.pop();
    }
    dagreGraph.setNode(node.id, {
      node,
      type: node.type,
      currentDepth: stack.length
    });
    if (stack.length > maxDepth) {
      maxDepth = stack.length;
    }
    if (node.type.startsWith('enter')) {
      stack.push(node.type);
    }
  });
  dagreGraph.nodes().map(id => ({
    id,
    node: dagreGraph.node(id)
  })).forEach(x => dagreGraph.setNode(x.id, {
    ...x.node,
    width: x.node.currentDepth === maxDepth ? baseWidth : baseWidth + (maxDepth - x.node.currentDepth) * 70,
    height: 50
  }));
  graph.edges().forEach(edge => {
    // Reverse source and destination for BT layout
    dagreGraph.setEdge(edge.w, edge.v, {
      type: 'workflowEdge'
    });
  });
  _dagre.default.layout(dagreGraph);
  const nodes = graph.nodes().map(id => {
    const dagreNode = dagreGraph.node(id);
    const graphNode = graph.node(id);
    return {
      id,
      data: {
        ...dagreNode,
        stepType: graphNode === null || graphNode === void 0 ? void 0 : graphNode.type,
        step: graphNode === null || graphNode === void 0 ? void 0 : graphNode.configuration,
        label: (graphNode === null || graphNode === void 0 ? void 0 : graphNode.label) || id
      },
      // See this: https://github.com/dagrejs/dagre/issues/287
      targetPosition: _react.Position.Bottom,
      // Reversed due to BT layout
      sourcePosition: _react.Position.Top,
      // Reversed due to BT layout
      style: {
        width: dagreNode.width,
        height: dagreNode.height
      },
      type: graphNode.type,
      position: {
        x: dagreNode.x - dagreNode.width / 2,
        y: dagreNode.y - dagreNode.height / 2
      }
    };
  });
  const edges = graph.edges().map(e => {
    var _graph$edge;
    return {
      id: `${e.v} -> ${e.w}`,
      source: e.v,
      target: e.w,
      label: (_graph$edge = graph.edge(e)) === null || _graph$edge === void 0 ? void 0 : _graph$edge.label
    };
  });
  return {
    nodes,
    edges
  };
}