"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkflowTraceSearch = useWorkflowTraceSearch;
var _react = require("react");
var _public = require("@kbn/kibana-react-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// APM API response structure for root transaction

function useWorkflowTraceSearch({
  workflowExecution
}) {
  const [result, setResult] = (0, _react.useState)({
    traceId: null,
    entryTransactionId: null,
    loading: false,
    error: null,
    isTraceComplete: false,
    expectedSpanCount: null,
    actualSpanCount: null
  });
  const {
    services
  } = (0, _public.useKibana)();
  (0, _react.useEffect)(() => {
    // Return early if workflowExecution is null (APM disabled) or has no traceId
    if (!(workflowExecution !== null && workflowExecution !== void 0 && workflowExecution.traceId)) {
      setResult({
        traceId: null,
        entryTransactionId: null,
        loading: false,
        error: workflowExecution === null ? null : new Error('No trace ID found for this workflow execution'),
        isTraceComplete: false,
        expectedSpanCount: null,
        actualSpanCount: null
      });
      return;
    }

    // Check if we already have the entry transaction ID stored
    if (workflowExecution.entryTransactionId) {
      setResult({
        traceId: workflowExecution.traceId,
        entryTransactionId: workflowExecution.entryTransactionId,
        loading: false,
        error: null,
        isTraceComplete: true,
        // Assume complete if we have stored data
        expectedSpanCount: null,
        actualSpanCount: null
      });
      return;
    }
    setResult(prev => ({
      ...prev,
      loading: true,
      error: null
    }));

    // Find the root transaction for this trace
    const fetchRootTransaction = async () => {
      try {
        var _services$http, _response$transaction;
        // Calculate time range for the search (expanded to include parent transaction)
        const startedAt = new Date(workflowExecution.startedAt);
        const finishedAt = workflowExecution.finishedAt ? new Date(workflowExecution.finishedAt) : new Date();
        const expandedStartTime = new Date(startedAt.getTime() - 30000); // 30 seconds before
        const expandedEndTime = new Date(finishedAt.getTime() + 30000); // 30 seconds after

        const response = await ((_services$http = services.http) === null || _services$http === void 0 ? void 0 : _services$http.get(`/internal/apm/traces/${workflowExecution.traceId}/root_transaction`, {
          query: {
            start: expandedStartTime.toISOString(),
            end: expandedEndTime.toISOString()
          }
        }));
        if (response !== null && response !== void 0 && (_response$transaction = response.transaction) !== null && _response$transaction !== void 0 && _response$transaction.id) {
          setResult({
            traceId: workflowExecution.traceId,
            entryTransactionId: response.transaction.id,
            loading: false,
            error: null,
            isTraceComplete: true,
            expectedSpanCount: null,
            actualSpanCount: null
          });
        } else {
          setResult({
            traceId: workflowExecution.traceId,
            entryTransactionId: workflowExecution.traceId,
            // Fallback to trace ID
            loading: false,
            error: null,
            isTraceComplete: true,
            expectedSpanCount: null,
            actualSpanCount: null
          });
        }
      } catch (error) {
        // eslint-disable-next-line no-console
        console.error('Error fetching root transaction:', error);
        // Fallback: use the stored trace ID without entry transaction
        setResult({
          traceId: workflowExecution.traceId,
          entryTransactionId: workflowExecution.traceId,
          // Fallback to trace ID
          loading: false,
          error: null,
          // Don't treat this as an error since we have the trace ID
          isTraceComplete: true,
          expectedSpanCount: null,
          actualSpanCount: null
        });
      }
    };
    fetchRootTransaction();
  }, [workflowExecution, services.http]);
  return result;
}