"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkflowUrlState = useWorkflowUrlState;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _queryString = require("query-string");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function useWorkflowUrlState() {
  const history = (0, _reactRouterDom.useHistory)();
  const location = (0, _reactRouterDom.useLocation)();
  const urlState = (0, _react.useMemo)(() => {
    const params = (0, _queryString.parse)(location.search);
    return {
      tab: params.tab || 'workflow',
      executionId: params.executionId,
      stepExecutionId: params.stepExecutionId,
      stepId: params.stepId
    };
  }, [location.search]);
  const updateUrlState = (0, _react.useCallback)(updates => {
    const currentParams = (0, _queryString.parse)(location.search);

    // Update the params with new values
    const newParams = {
      ...currentParams,
      ...updates
    };

    // Remove undefined values to keep URL clean
    const cleanParams = {};
    Object.entries(newParams).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        cleanParams[key] = value;
      }
    });

    // Update the URL without causing a full page reload
    const newSearch = (0, _queryString.stringify)(cleanParams, {
      encode: false
    });
    const newLocation = {
      ...location,
      search: newSearch ? `?${newSearch}` : ''
    };
    history.replace(newLocation);
  }, [history, location]);
  const setActiveTab = (0, _react.useCallback)(tab => {
    // When switching to other tab, clear execution selection
    updateUrlState({
      executionId: undefined,
      stepExecutionId: undefined,
      stepId: undefined,
      tab
    });
  }, [updateUrlState]);
  const setSelectedExecution = (0, _react.useCallback)(executionId => {
    updateUrlState({
      executionId: executionId || undefined,
      stepExecutionId: undefined,
      stepId: undefined
    });
  }, [updateUrlState]);
  const setSelectedStepExecution = (0, _react.useCallback)(stepExecutionId => {
    updateUrlState({
      stepExecutionId: stepExecutionId || undefined,
      stepId: undefined
    });
  }, [updateUrlState]);
  const setSelectedStep = (0, _react.useCallback)(stepId => {
    updateUrlState({
      stepId: stepId || undefined
    });
  }, [updateUrlState]);
  return {
    // Current state
    activeTab: urlState.tab,
    selectedExecutionId: urlState.executionId,
    selectedStepExecutionId: urlState.stepExecutionId,
    selectedStepId: urlState.stepId,
    // State setters
    setActiveTab,
    setSelectedExecution,
    setSelectedStepExecution,
    setSelectedStep,
    updateUrlState
  };
}