"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkflowUrlStateAdvanced = useWorkflowUrlStateAdvanced;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _public = require("@kbn/kibana-utils-plugin/public");
var _public2 = require("@kbn/kibana-react-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const defaultState = {
  tab: 'workflow'
};

/**
 * Advanced URL state management using Kibana's KbnUrlStateStorage
 * This approach is better for complex state with nested objects, arrays, or when you need:
 * - State hashing to avoid long URLs
 * - Better serialization (Rison encoding)
 * - Integration with Kibana's global state management
 */
function useWorkflowUrlStateAdvanced() {
  const history = (0, _reactRouterDom.useHistory)();
  const {
    notifications,
    uiSettings
  } = (0, _public2.useKibana)().services;

  // Create URL state storage with Kibana's utilities
  const [urlStateStorage] = (0, _react.useState)(() => (0, _public.createKbnUrlStateStorage)({
    useHash: Boolean(uiSettings === null || uiSettings === void 0 ? void 0 : uiSettings.get('state:storeInSessionStorage')),
    history,
    ...(notifications !== null && notifications !== void 0 && notifications.toasts ? (0, _public.withNotifyOnErrors)(notifications.toasts) : {})
  }));

  // Create state container
  const [stateContainer] = (0, _react.useState)(() => (0, _public.createStateContainer)(
  // Get initial state from URL or use default
  urlStateStorage.get('_w') || defaultState));

  // Set up sync between state container and URL
  (0, _react.useEffect)(() => {
    const {
      start,
      stop
    } = (0, _public.syncState)({
      storageKey: '_w',
      // URL key for workflow state
      stateContainer: {
        ...stateContainer,
        set: state => {
          if (state) {
            stateContainer.set(state);
          }
        }
      },
      stateStorage: urlStateStorage
    });
    start();
    return stop;
  }, [stateContainer, urlStateStorage]);

  // Get current state
  const state = stateContainer.get();

  // Helper functions to update specific parts of state
  const setActiveTab = (0, _react.useCallback)(tab => {
    const updates = {
      tab
    };

    // When switching to workflow tab, clear execution-specific state
    if (tab === 'workflow') {
      updates.executionId = undefined;
      updates.executionFilters = undefined;
    }
    stateContainer.set({
      ...state,
      ...updates
    });
  }, [state, stateContainer]);
  const setSelectedExecution = (0, _react.useCallback)(executionId => {
    stateContainer.set({
      ...state,
      tab: 'executions',
      // Automatically switch to executions tab
      executionId: executionId || undefined
    });
  }, [state, stateContainer]);
  const setExecutionFilters = (0, _react.useCallback)(filters => {
    stateContainer.set({
      ...state,
      executionFilters: filters
    });
  }, [state, stateContainer]);
  const updateState = (0, _react.useCallback)(updates => {
    stateContainer.set({
      ...state,
      ...updates
    });
  }, [state, stateContainer]);
  return {
    // Current state
    activeTab: state.tab,
    selectedExecutionId: state.executionId,
    executionFilters: state.executionFilters,
    // State setters
    setActiveTab,
    setSelectedExecution,
    setExecutionFilters,
    updateState,
    // Full state for advanced usage
    state,
    stateContainer
  };
}