"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.COPY_AS_LANGUAGES = void 0;
exports.convertStepToLanguage = convertStepToLanguage;
exports.copyAsConsole = copyAsConsole;
exports.copyStepAs = copyStepAs;
exports.copyToClipboard = copyToClipboard;
exports.getConsoleFormat = getConsoleFormat;
exports.getCurlCommand = getCurlCommand;
var _i18n = require("@kbn/i18n");
var _elasticsearch_step_utils = require("./elasticsearch_step_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Language options for "Copy as" functionality
 */
const COPY_AS_LANGUAGES = exports.COPY_AS_LANGUAGES = [{
  value: 'curl',
  label: 'cURL'
}, {
  value: 'javascript',
  label: 'JavaScript'
}, {
  value: 'python',
  label: 'Python'
}, {
  value: 'java',
  label: 'Java'
}, {
  value: 'csharp',
  label: 'C#'
}, {
  value: 'php',
  label: 'PHP'
}, {
  value: 'go',
  label: 'Go'
}];
/**
 * Converts an Elasticsearch workflow step to cURL command
 */
function getCurlCommand(step, elasticsearchBaseUrl = 'http://localhost:9200') {
  const request = (0, _elasticsearch_step_utils.stepToConsoleRequest)(step);
  const curlUrl = `${elasticsearchBaseUrl}${request.url}`;
  let curlCommand = `curl -X${request.method} "${curlUrl}"`;
  if (request.data && request.data.length > 0) {
    const joinedData = request.data.join('\n');
    curlCommand += ` -H "Content-Type: application/json" -d'${joinedData}'`;
  }
  return curlCommand;
}

/**
 * Converts an Elasticsearch workflow step to Console format
 */
function getConsoleFormat(step) {
  const request = (0, _elasticsearch_step_utils.stepToConsoleRequest)(step);
  let consoleFormat = `${request.method} ${request.url}`;
  if (request.data && request.data.length > 0) {
    consoleFormat += '\n' + request.data.join('\n');
  }
  return consoleFormat;
}

/**
 * Copies text to clipboard
 */
async function copyToClipboard(text) {
  try {
    if (navigator.clipboard && window.isSecureContext) {
      await navigator.clipboard.writeText(text);
      return true;
    } else {
      // Fallback for older browsers
      const textArea = document.createElement('textarea');
      textArea.value = text;
      textArea.style.position = 'absolute';
      textArea.style.left = '-999999px';
      document.body.appendChild(textArea);
      textArea.select();
      document.execCommand('copy');
      document.body.removeChild(textArea);
      return true;
    }
  } catch (error) {
    // console.error('Failed to copy to clipboard:', error);
    return false;
  }
}

/**
 * Converts a workflow step to the specified language format using the Console API
 */
async function convertStepToLanguage(step, language, options) {
  const {
    http,
    esHost = 'http://localhost:9200',
    kibanaHost = window.location.origin
  } = options;
  const request = (0, _elasticsearch_step_utils.stepToConsoleRequest)(step);
  try {
    const response = await http.post('/api/console/convert_request_to_language', {
      query: {
        language,
        esHost,
        kibanaHost
      },
      body: JSON.stringify([request]),
      headers: {
        'Content-Type': 'application/json'
      }
    });
    return {
      data: response
    };
  } catch (error) {
    return {
      error: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.conversionError', {
        defaultMessage: 'Failed to convert request to {language}',
        values: {
          language
        }
      })
    };
  }
}

/**
 * Copies a workflow step as the specified language format
 */
async function copyStepAs(step, language, options) {
  const {
    notifications
  } = options;
  let textToCopy;
  let success = false;
  try {
    if (language === 'curl') {
      // For cURL, use our local implementation for better performance
      textToCopy = getCurlCommand(step, options.esHost);
      success = await copyToClipboard(textToCopy);
    } else {
      // For other languages, use the Console API
      const {
        data,
        error
      } = await convertStepToLanguage(step, language, options);
      if (error) {
        notifications.toasts.addDanger({
          title: error
        });
        return;
      }
      if (data) {
        textToCopy = data;
        success = await copyToClipboard(textToCopy);
      }
    }
    if (success) {
      var _COPY_AS_LANGUAGES$fi;
      const languageLabel = ((_COPY_AS_LANGUAGES$fi = COPY_AS_LANGUAGES.find(l => l.value === language)) === null || _COPY_AS_LANGUAGES$fi === void 0 ? void 0 : _COPY_AS_LANGUAGES$fi.label) || language;
      notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.success', {
          defaultMessage: 'Copied as {language}',
          values: {
            language: languageLabel
          }
        })
      });
    } else {
      notifications.toasts.addDanger({
        title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.clipboardError', {
          defaultMessage: 'Failed to copy to clipboard'
        })
      });
    }
  } catch (error) {
    notifications.toasts.addDanger({
      title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.unexpectedError', {
        defaultMessage: 'An unexpected error occurred'
      })
    });
  }
}

/**
 * Copies a workflow step in Console format
 */
async function copyAsConsole(step, options) {
  const {
    notifications
  } = options;
  try {
    const consoleFormat = getConsoleFormat(step);
    const success = await copyToClipboard(consoleFormat);
    if (success) {
      notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.consoleSuccess', {
          defaultMessage: 'Copied as Console format'
        })
      });
    } else {
      notifications.toasts.addDanger({
        title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.clipboardError', {
          defaultMessage: 'Failed to copy to clipboard'
        })
      });
    }
  } catch (error) {
    notifications.toasts.addDanger({
      title: _i18n.i18n.translate('workflows.workflowDetail.yamlEditor.copyAs.unexpectedError', {
        defaultMessage: 'An unexpected error occurred'
      })
    });
  }
}