"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BaseMonacoConnectorHandler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Abstract base class for Monaco connector handlers
 * Provides common functionality for Monaco editor extensions that handle specific connector types
 */
class BaseMonacoConnectorHandler {
  constructor(name, priority, supportedPrefixes) {
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "priority", void 0);
    (0, _defineProperty2.default)(this, "supportedPrefixes", void 0);
    this.name = name;
    this.priority = priority;
    this.supportedPrefixes = supportedPrefixes;
  }

  /**
   * Check if this handler can process the given connector type
   */
  canHandle(connectorType) {
    return this.supportedPrefixes.some(prefix => connectorType.startsWith(prefix));
  }

  /**
   * Get priority for this handler (higher = more specific, handled first)
   */
  getPriority() {
    return this.priority;
  }

  /**
   * Generate hover content for the connector - must be implemented by subclasses
   */

  /**
   * Generate floating action buttons for the connector - must be implemented by subclasses
   */

  /**
   * Get examples for the connector type - must be implemented by subclasses
   */

  /**
   * Helper method to create consistent markdown content
   */
  createMarkdownContent(content) {
    return {
      value: content,
      isTrusted: true,
      supportHtml: true
    };
  }

  /**
   * Helper method to create action info objects
   */
  createActionInfo(id, label, handler, options = {}) {
    return {
      id,
      label,
      handler,
      icon: options.icon,
      tooltip: options.tooltip || label,
      priority: options.priority || 0
    };
  }

  /**
   * Helper method to check if we're in a specific YAML context
   */
  isInContext(context, expectedPath) {
    if (context.yamlPath.length < expectedPath.length) {
      return false;
    }
    for (let i = 0; i < expectedPath.length; i++) {
      if (context.yamlPath[i] !== expectedPath[i]) {
        return false;
      }
    }
    return true;
  }

  /**
   * Helper method to extract step information from context
   */
  getStepInfo(context) {
    const {
      stepContext
    } = context;
    return {
      stepName: stepContext === null || stepContext === void 0 ? void 0 : stepContext.stepName,
      stepType: stepContext === null || stepContext === void 0 ? void 0 : stepContext.stepType,
      isInWithBlock: (stepContext === null || stepContext === void 0 ? void 0 : stepContext.isInWithBlock) || false
    };
  }

  /**
   * Helper method to format connector type for display
   */
  formatConnectorType(connectorType) {
    // Remove prefixes for cleaner display
    return connectorType.replace(/^elasticsearch\./, '').replace(/^kibana\./, '').replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
  }

  /**
   * Helper method to create parameter documentation
   */
  createParameterDocumentation(paramName, paramType, description, examples) {
    const lines = [`**Parameter**: \`${paramName}\` (${paramType})`];
    if (description) {
      lines.push('', description);
    }
    if (examples && examples.length > 0) {
      lines.push('', '**Examples:**');
      examples.slice(0, 3).forEach(example => {
        if (typeof example === 'string') {
          lines.push(`- \`${example}\``);
        } else {
          lines.push(`- \`${JSON.stringify(example)}\``);
        }
      });
    }
    return lines.join('\n');
  }

  /**
   * Helper method to create connector overview documentation
   */
  createConnectorOverview(connectorType, description, additionalInfo) {
    const lines = [`**Connector**: \`${connectorType}\``, '', description];
    if (additionalInfo && additionalInfo.length > 0) {
      lines.push('', ...additionalInfo);
    }
    return lines.join('\n');
  }
}
exports.BaseMonacoConnectorHandler = BaseMonacoConnectorHandler;