"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.insertStepSnippet = insertStepSnippet;
var _monaco = require("@kbn/monaco");
var _workflows = require("@kbn/workflows");
var _yaml_utils = require("../../../../../common/lib/yaml_utils");
var _utils = require("../utils");
var _get_indent_level = require("../get_indent_level");
var _prepend_indent_to_lines = require("../prepend_indent_to_lines");
var _generate_builtin_step_snippet = require("./generate_builtin_step_snippet");
var _generate_connector_snippet = require("./generate_connector_snippet");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Algorithm:
// 1. If no cursor position is provided, find the next line after the last step node in root range
// 2. If cursor position is provided, get next line after the end of step node nearest to the cursor (including nested steps, e.g. foreach, if, etc.)
// 3. If no step nodes found, add "steps:" section in the first line of yaml

function insertStepSnippet(model, yamlDocument, stepType, cursorPosition, editor) {
  let snippetText = '';
  // we need it to be 1-indexed
  const lastLineNumber = model.getLineCount();
  let insertStepsSection = false;
  // by default, insert at line after the last line of the yaml file
  let insertAtLineNumber = lastLineNumber + 1;
  const stepNodes = (0, _yaml_utils.getStepNodesWithType)(yamlDocument);
  let nearestStepNode = null;
  if (cursorPosition) {
    const absolutePosition = model.getOffsetAt(cursorPosition);
    nearestStepNode = (0, _yaml_utils.getStepNodeAtPosition)(yamlDocument, absolutePosition);
  }
  const stepNode = nearestStepNode || (stepNodes.length > 0 ? stepNodes[stepNodes.length - 1] : null);
  let indentLevel = 0;
  if (!stepNode) {
    insertStepsSection = true;
    indentLevel = 2;
  } else {
    const stepRange = (0, _utils.getMonacoRangeFromYamlNode)(model, stepNode);
    if (stepRange) {
      insertAtLineNumber = stepRange.endLineNumber === lastLineNumber ? lastLineNumber + 1 : stepRange.endLineNumber;
      // get indent from the first line of the step node
      indentLevel = (0, _get_indent_level.getIndentLevelFromLineNumber)(model, stepRange.startLineNumber);
    }
  }
  if ((0, _workflows.isBuiltInStepType)(stepType)) {
    snippetText = (0, _generate_builtin_step_snippet.generateBuiltInStepSnippet)(stepType, {
      full: true,
      withStepsSection: insertStepsSection
    });
  } else {
    snippetText = (0, _generate_connector_snippet.generateConnectorSnippet)(stepType, {
      full: true,
      withStepsSection: insertStepsSection
    });
  }

  // Create separate undo boundary for each snippet insertion
  if (editor) {
    editor.pushUndoStop();
  }
  model.pushEditOperations(null, [{
    range: new _monaco.monaco.Range(insertAtLineNumber, 1, insertAtLineNumber, 1),
    // if we are inserting the steps section, we don't need to indent the snippet, the step is at root level
    text: insertStepsSection ? snippetText : (0, _prepend_indent_to_lines.prependIndentToLines)(snippetText, indentLevel)
  }], () => null);
  if (editor) {
    editor.pushUndoStop();
  }
}