"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowsPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("../common");
var _workflow_task_runner = require("./tasks/workflow_task_runner");
var _workflow_task_scheduler = require("./tasks/workflow_task_scheduler");
var _workflows_management_api = require("./workflows_management/workflows_management_api");
var _workflows_management_routes = require("./workflows_management/workflows_management_routes");
var _workflows_management_service = require("./workflows_management/workflows_management_service");
var _workflows = require("./connectors/workflows");
var _features = require("./features");
var _ui_settings = require("./ui_settings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Import the workflows connector

class WorkflowsPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "workflowsService", null);
    (0, _defineProperty2.default)(this, "workflowTaskScheduler", null);
    (0, _defineProperty2.default)(this, "unsecureActionsClient", null);
    (0, _defineProperty2.default)(this, "api", null);
    (0, _defineProperty2.default)(this, "spaces", null);
    this.logger = initializerContext.logger.get();
    this.config = initializerContext.config.get();
  }
  setup(core, plugins) {
    var _plugins$spaces;
    this.logger.debug('Workflows Management: Setup');
    (0, _ui_settings.registerUISettings)({
      uiSettings: core.uiSettings
    });

    // Register workflows connector if actions plugin is available
    if (plugins.actions) {
      // Create workflows service function for the connector
      const getWorkflowsService = async request => {
        // Return a function that will be called by the connector
        return async (workflowId, spaceId, inputs) => {
          if (!this.api) {
            throw new Error('Workflows management API not initialized');
          }

          // Get the workflow first
          const workflow = await this.api.getWorkflow(workflowId, spaceId);
          if (!workflow) {
            throw new Error(`Workflow not found: ${workflowId}`);
          }
          if (!workflow.definition) {
            throw new Error(`Workflow definition not found: ${workflowId}`);
          }
          if (!workflow.valid) {
            throw new Error(`Workflow is not valid: ${workflowId}`);
          }
          const workflowToRun = {
            id: workflow.id,
            name: workflow.name,
            enabled: workflow.enabled,
            definition: workflow.definition,
            yaml: workflow.yaml
          };

          // Run the workflow, @tb: maybe switch to scheduler?
          return await this.api.runWorkflow(workflowToRun, spaceId, inputs, request);
        };
      };

      // Register the workflows connector
      plugins.actions.registerType((0, _workflows.getConnectorType)({
        getWorkflowsService
      }));

      // Register connector adapter for alerting if available
      if (plugins.alerting) {
        plugins.alerting.registerConnectorAdapter((0, _workflows.getWorkflowsConnectorAdapter)());
      }
    }

    // Register workflow task definition
    if (plugins.taskManager) {
      plugins.taskManager.registerTaskDefinitions({
        'workflow:scheduled': {
          title: 'Scheduled Workflow Execution',
          description: 'Executes workflows on a scheduled basis',
          timeout: '5m',
          maxAttempts: 3,
          createTaskRunner: ({
            taskInstance,
            fakeRequest
          }) => {
            // Capture the plugin instance in a closure
            const plugin = this;
            // Use a factory pattern to get dependencies when the task runs
            return {
              async run() {
                // Get dependencies when the task actually runs
                const [, pluginsStart] = await core.getStartServices();

                // Create the actual task runner with dependencies
                const taskRunner = (0, _workflow_task_runner.createWorkflowTaskRunner)({
                  logger: plugin.logger,
                  workflowsService: plugin.workflowsService,
                  workflowsExecutionEngine: pluginsStart.workflowsExecutionEngine,
                  actionsClient: plugin.unsecureActionsClient
                })({
                  taskInstance,
                  fakeRequest
                });
                return taskRunner.run();
              },
              async cancel() {
                // Cancel function for the task
              }
            };
          }
        }
      });
    }

    // Register saved object types

    (0, _features.registerFeatures)(plugins);
    this.logger.debug('Workflows Management: Creating router');
    const router = core.http.createRouter();
    this.logger.debug('Workflows Management: Creating workflows service');

    // Get ES client from core
    const esClientPromise = core.getStartServices().then(([coreStart]) => coreStart.elasticsearch.client.asInternalUser);
    const getWorkflowExecutionEngine = () => core.getStartServices().then(([, pluginsStart]) => pluginsStart.workflowsExecutionEngine);
    this.workflowsService = new _workflows_management_service.WorkflowsService(esClientPromise, this.logger, _common.WORKFLOWS_EXECUTIONS_INDEX, _common.WORKFLOWS_STEP_EXECUTIONS_INDEX, _common.WORKFLOWS_EXECUTION_LOGS_INDEX, this.config.logging.console);
    this.api = new _workflows_management_api.WorkflowsManagementApi(this.workflowsService, getWorkflowExecutionEngine);
    this.spaces = (_plugins$spaces = plugins.spaces) === null || _plugins$spaces === void 0 ? void 0 : _plugins$spaces.spacesService;

    // Register server side APIs
    (0, _workflows_management_routes.defineRoutes)(router, this.api, this.logger, this.spaces);
    return {
      management: this.api
    };
  }
  start(core, plugins) {
    this.logger.info('Workflows Management: Start');
    this.unsecureActionsClient = plugins.actions.getUnsecuredActionsClient();

    // Initialize workflow task scheduler with the start contract
    this.workflowTaskScheduler = new _workflow_task_scheduler.WorkflowTaskScheduler(this.logger, plugins.taskManager);

    // Set task scheduler and security service in workflows service
    if (this.workflowsService) {
      this.workflowsService.setTaskScheduler(this.workflowTaskScheduler);
      if (plugins.security) {
        this.workflowsService.setSecurityService(core.security);
      }
    }
    const actionsTypes = plugins.actions.getAllTypes();
    this.logger.debug(`Available action types: ${actionsTypes.join(', ')}`);
    this.logger.info('Workflows Management: Started');
    return {};
  }
  stop() {}
}
exports.WorkflowsPlugin = WorkflowsPlugin;