"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SimpleWorkflowLogger = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Simple interfaces for workflow logging

// Simple logger implementation with console support and log search capabilities
class SimpleWorkflowLogger {
  constructor(logger, esClient, logsIndex, enableConsoleLogging = false) {
    this.logger = logger;
    this.esClient = esClient;
    this.logsIndex = logsIndex;
    this.enableConsoleLogging = enableConsoleLogging;
  }
  logInfo(message, meta) {
    if (this.enableConsoleLogging) {
      this.logger.info(`🔄 WORKFLOW: ${message}`, meta);
    }
  }
  logError(message, error, meta) {
    if (this.enableConsoleLogging) {
      this.logger.error(`🔄 WORKFLOW: ${message}`, {
        error,
        ...meta
      });
    }
  }
  async searchLogs(params, spaceId) {
    try {
      var _response$hits$total;
      const {
        limit = 100,
        offset = 0,
        sortField = '@timestamp',
        sortOrder = 'desc'
      } = params;

      // Map API field names to Elasticsearch field names
      const fieldMapping = {
        timestamp: '@timestamp',
        '@timestamp': '@timestamp'
      };
      const mappedSortField = fieldMapping[sortField] || sortField;
      const mustQueries = [];
      if ('executionId' in params) {
        mustQueries.push({
          term: {
            'workflow.execution_id.keyword': params.executionId
          }
        });
      }
      if ('stepExecutionId' in params && params.stepExecutionId) {
        mustQueries.push({
          term: {
            'workflow.step_execution_id.keyword': params.stepExecutionId
          }
        });
      }
      if ('stepId' in params && params.stepId) {
        mustQueries.push({
          term: {
            'workflow.step_id.keyword': params.stepId
          }
        });
      }
      if (spaceId) {
        mustQueries.push({
          term: {
            'spaceId.keyword': spaceId
          }
        });
      }
      const response = await this.esClient.search({
        index: this.logsIndex,
        size: limit,
        from: offset,
        query: {
          bool: {
            must: mustQueries
          }
        },
        sort: [{
          [mappedSortField]: {
            order: sortOrder
          }
        }]
      });
      const logs = response.hits.hits.map(hit => hit._source);
      const total = typeof response.hits.total === 'number' ? response.hits.total : ((_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) || 0;
      return {
        total,
        logs
      };
    } catch (error) {
      this.logger.error('Failed to search workflow logs', error);
      return {
        total: 0,
        logs: []
      };
    }
  }
}
exports.SimpleWorkflowLogger = SimpleWorkflowLogger;