"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WorkflowsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _workflows = require("@kbn/workflows");
var _uuid = require("uuid");
var _errors = require("../../common/lib/errors");
var _validate_step_names = require("../../common/lib/validate_step_names");
var _yaml_utils = require("../../common/lib/yaml_utils");
var _schema = require("../../common/schema");
var _get_user = require("../lib/get_user");
var _schedule_utils = require("../lib/schedule_utils");
var _workflow_storage = require("../storage/workflow_storage");
var _create_index = require("./lib/create_index");
var _get_workflow_execution = require("./lib/get_workflow_execution");
var _index_mappings = require("./lib/index_mappings");
var _search_step_executions = require("./lib/search_step_executions");
var _search_workflow_executions = require("./lib/search_workflow_executions");
var _workflow_logger = require("./lib/workflow_logger");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class WorkflowsService {
  constructor(esClientPromise, logger, workflowsExecutionIndex, stepsExecutionIndex, workflowExecutionLogsIndex, enableConsoleLogging = false) {
    (0, _defineProperty2.default)(this, "esClient", null);
    (0, _defineProperty2.default)(this, "workflowStorage", null);
    (0, _defineProperty2.default)(this, "taskScheduler", null);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "workflowsExecutionIndex", void 0);
    (0, _defineProperty2.default)(this, "stepsExecutionIndex", void 0);
    (0, _defineProperty2.default)(this, "workflowEventLoggerService", null);
    (0, _defineProperty2.default)(this, "security", void 0);
    this.logger = logger;
    this.stepsExecutionIndex = stepsExecutionIndex;
    this.workflowsExecutionIndex = workflowsExecutionIndex;
    void this.initialize(esClientPromise, workflowExecutionLogsIndex, enableConsoleLogging);
  }
  setTaskScheduler(taskScheduler) {
    this.taskScheduler = taskScheduler;
  }
  setSecurityService(security) {
    this.security = security;
  }
  async initialize(esClientPromise, workflowExecutionLogsIndex, enableConsoleLogging = false) {
    this.esClient = await esClientPromise;

    // Initialize workflow storage
    this.workflowStorage = (0, _workflow_storage.createStorage)({
      logger: this.logger,
      esClient: this.esClient
    });
    this.workflowEventLoggerService = new _workflow_logger.SimpleWorkflowLogger(this.logger, this.esClient, workflowExecutionLogsIndex, enableConsoleLogging);

    // Create execution indices
    await (0, _create_index.createOrUpdateIndex)({
      esClient: this.esClient,
      indexName: this.workflowsExecutionIndex,
      mappings: _index_mappings.WORKFLOWS_EXECUTIONS_INDEX_MAPPINGS,
      logger: this.logger
    });
    await (0, _create_index.createOrUpdateIndex)({
      esClient: this.esClient,
      indexName: this.stepsExecutionIndex,
      mappings: _index_mappings.WORKFLOWS_STEP_EXECUTIONS_INDEX_MAPPINGS,
      logger: this.logger
    });
  }
  async getWorkflow(id, spaceId) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    try {
      const response = await this.workflowStorage.getClient().search({
        query: {
          bool: {
            must: [{
              ids: {
                values: [id]
              }
            }, {
              term: {
                spaceId
              }
            }]
          }
        },
        size: 1,
        track_total_hits: false
      });
      if (response.hits.hits.length === 0) {
        return null;
      }
      const document = response.hits.hits[0];
      return this.transformStorageDocumentToWorkflowDto(document._id, document._source);
    } catch (error) {
      if (error.statusCode === 404) {
        return null;
      }
      throw error;
    }
  }
  async createWorkflow(workflow, spaceId, request) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    const parsedYaml = (0, _yaml_utils.parseWorkflowYamlToJSON)(workflow.yaml, (0, _schema.getWorkflowZodSchemaLoose)());
    if (!parsedYaml.success) {
      throw new Error('Invalid workflow yaml: ' + parsedYaml.error.message);
    }

    // Validate step name uniqueness
    const stepValidation = (0, _validate_step_names.validateStepNameUniqueness)(parsedYaml.data);
    if (!stepValidation.isValid) {
      const errorMessages = stepValidation.errors.map(error => error.message);
      throw new _errors.WorkflowValidationError('Workflow validation failed: Step names must be unique throughout the workflow.', errorMessages);
    }

    // The type of parsedYaml.data is validated by getWorkflowZodSchemaLoose(), so this assertion is partially safe.
    const workflowToCreate = (0, _workflows.transformWorkflowYamlJsontoEsWorkflow)(parsedYaml.data);
    const authenticatedUser = (0, _get_user.getAuthenticatedUser)(request, this.security);
    const now = new Date();
    const workflowData = {
      name: workflowToCreate.name,
      description: workflowToCreate.description,
      enabled: workflowToCreate.enabled,
      tags: workflowToCreate.tags || [],
      yaml: workflow.yaml,
      definition: workflowToCreate.definition,
      createdBy: authenticatedUser,
      lastUpdatedBy: authenticatedUser,
      spaceId,
      valid: true,
      deleted_at: null,
      created_at: now.toISOString(),
      updated_at: now.toISOString()
    };
    const id = this.generateWorkflowId();
    await this.workflowStorage.getClient().index({
      id,
      document: workflowData
    });

    // Schedule the workflow if it has triggers
    if (this.taskScheduler && workflowToCreate.definition.triggers) {
      for (const trigger of workflowToCreate.definition.triggers) {
        if (trigger.type === 'scheduled' && trigger.enabled) {
          await this.taskScheduler.scheduleWorkflowTask(id, spaceId, trigger, request);
        }
      }
    }
    return this.transformStorageDocumentToWorkflowDto(id, workflowData);
  }
  async updateWorkflow(id, workflow, spaceId, request) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    try {
      // First check if the workflow exists and belongs to the correct space
      const searchResponse = await this.workflowStorage.getClient().search({
        query: {
          bool: {
            must: [{
              ids: {
                values: [id]
              }
            }, {
              term: {
                spaceId
              }
            }]
          }
        },
        size: 1,
        track_total_hits: false
      });
      if (searchResponse.hits.hits.length === 0) {
        throw new Error(`Workflow with id ${id} not found in space ${spaceId}`);
      }
      const existingDocument = searchResponse.hits.hits[0];
      if (!existingDocument._source) {
        throw new Error(`Workflow with id ${id} not found`);
      }
      const authenticatedUser = (0, _get_user.getAuthenticatedUser)(request, this.security);
      const now = new Date();
      const validationErrors = [];
      const updatedData = {
        lastUpdatedBy: authenticatedUser,
        updated_at: now.toISOString()
      };

      // If yaml is being updated, validate and update definition
      let shouldUpdateScheduler = false;

      // Check if enabled state is being changed
      if (workflow.enabled !== undefined && workflow.enabled !== existingDocument._source.enabled) {
        shouldUpdateScheduler = true;
      }

      // Handle yaml updates - this will also update definition and validation
      if (workflow.yaml) {
        // we always update the yaml, even if it's not valid, to allow users to save draft
        updatedData.yaml = workflow.yaml;
        const parsedYaml = (0, _yaml_utils.parseWorkflowYamlToJSON)(workflow.yaml, (0, _schema.getWorkflowZodSchema)());
        if (!parsedYaml.success) {
          updatedData.definition = undefined;
          updatedData.enabled = false;
          updatedData.valid = false;
          if (parsedYaml.error instanceof _errors.InvalidYamlSchemaError && parsedYaml.error.formattedZodError) {
            validationErrors.push(...parsedYaml.error.formattedZodError.issues.map(error => error.message));
          } else {
            validationErrors.push(parsedYaml.error.message);
          }
          shouldUpdateScheduler = true;
        } else {
          // Validate step name uniqueness
          const stepValidation = (0, _validate_step_names.validateStepNameUniqueness)(parsedYaml.data);
          if (!stepValidation.isValid) {
            updatedData.definition = undefined;
            updatedData.enabled = false;
            updatedData.valid = false;
            validationErrors.push(...stepValidation.errors.map(error => error.message));
            shouldUpdateScheduler = true;
          } else {
            const workflowDef = (0, _workflows.transformWorkflowYamlJsontoEsWorkflow)(parsedYaml.data);
            // Update all fields from the transformed YAML, not just definition
            updatedData.definition = workflowDef.definition;
            updatedData.name = workflowDef.name;
            updatedData.enabled = workflowDef.enabled;
            updatedData.description = workflowDef.description;
            updatedData.tags = workflowDef.tags;
            updatedData.valid = true;
            updatedData.yaml = workflow.yaml;
            shouldUpdateScheduler = true;
          }
        }
      }

      // Handle individual field updates only when YAML is not being updated
      if (!workflow.yaml) {
        var _existingDocument$_so2;
        let yamlUpdated = false;
        if (workflow.name !== undefined) {
          updatedData.name = workflow.name;
          yamlUpdated = true;
        }
        if (workflow.enabled !== undefined) {
          var _existingDocument$_so;
          // If enabling a workflow, ensure it has a valid definition
          if (workflow.enabled && (_existingDocument$_so = existingDocument._source) !== null && _existingDocument$_so !== void 0 && _existingDocument$_so.definition) {
            updatedData.enabled = true;
          } else if (!workflow.enabled) {
            updatedData.enabled = false;
          }
          yamlUpdated = true;
        }
        if (workflow.description !== undefined) {
          updatedData.description = workflow.description;
          yamlUpdated = true;
        }
        if (workflow.tags !== undefined) {
          updatedData.tags = workflow.tags;
          yamlUpdated = true;
        }

        // If any individual fields were updated, regenerate the YAML content
        if (yamlUpdated && (_existingDocument$_so2 = existingDocument._source) !== null && _existingDocument$_so2 !== void 0 && _existingDocument$_so2.definition) {
          const updatedWorkflowDefinition = {
            ...existingDocument._source.definition,
            ...(workflow.name !== undefined && {
              name: workflow.name
            }),
            ...(workflow.enabled !== undefined && {
              enabled: updatedData.enabled
            }),
            ...(workflow.description !== undefined && {
              description: workflow.description
            }),
            ...(workflow.tags !== undefined && {
              tags: workflow.tags
            })
          };
          updatedData.yaml = (0, _yaml_utils.stringifyWorkflowDefinition)(updatedWorkflowDefinition);
        }
      }
      const finalData = {
        ...existingDocument._source,
        ...updatedData
      };
      await this.workflowStorage.getClient().index({
        id,
        document: finalData
      });

      // Update task scheduler if needed
      if (shouldUpdateScheduler && this.taskScheduler) {
        try {
          if (finalData.definition && finalData.valid && finalData.enabled) {
            // Check if workflow has scheduled triggers before updating scheduler
            const workflowHasScheduledTriggers = (0, _schedule_utils.hasScheduledTriggers)(finalData.definition.triggers || []);
            if (workflowHasScheduledTriggers) {
              // Get the updated workflow from storage
              const updatedWorkflow = await this.getWorkflow(id, spaceId);
              if (updatedWorkflow && updatedWorkflow.definition) {
                // Convert WorkflowDetailDto to EsWorkflow for scheduler
                const workflowForScheduler = {
                  ...updatedWorkflow,
                  definition: updatedWorkflow.definition,
                  // We already checked it's not null
                  tags: [],
                  // TODO: Add tags support to WorkflowDetailDto
                  deleted_at: null
                };
                await this.taskScheduler.updateWorkflowTasks(workflowForScheduler, spaceId, request);
                this.logger.info(`Updated scheduled tasks for workflow ${id}`);
              }
            } else {
              // No scheduled triggers, remove any existing scheduled tasks
              await this.taskScheduler.unscheduleWorkflowTasks(id);
              this.logger.info(`Removed scheduled tasks for workflow ${id} (no scheduled triggers)`);
            }
          } else {
            // If workflow is invalid or disabled, remove all scheduled tasks
            await this.taskScheduler.unscheduleWorkflowTasks(id);
            this.logger.info(`Removed all scheduled tasks for workflow ${id} (workflow disabled or invalid)`);
          }
        } catch (error) {
          this.logger.error(`Failed to update scheduled tasks for workflow ${id}: ${error}`);
          // Don't throw the error - the workflow update should succeed even if scheduler update fails
        }
      }
      return {
        id,
        lastUpdatedAt: new Date(finalData.updated_at),
        lastUpdatedBy: finalData.lastUpdatedBy,
        enabled: finalData.enabled,
        validationErrors,
        valid: finalData.valid
      };
    } catch (error) {
      if (error.statusCode === 404) {
        throw new Error(`Workflow with id ${id} not found`);
      }
      throw error;
    }
  }
  async deleteWorkflows(ids, spaceId) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    const now = new Date();

    // Soft delete by setting deleted_at timestamp
    for (const id of ids) {
      try {
        // Check if workflow exists and belongs to the correct space
        const searchResponse = await this.workflowStorage.getClient().search({
          query: {
            bool: {
              must: [{
                ids: {
                  values: [id]
                }
              }, {
                term: {
                  spaceId
                }
              }]
            }
          },
          size: 1,
          track_total_hits: false
        });
        if (searchResponse.hits.hits.length > 0) {
          const existingDocument = searchResponse.hits.hits[0];
          const updatedData = {
            ...existingDocument._source,
            deleted_at: now,
            enabled: false
          };
          await this.workflowStorage.getClient().index({
            id,
            document: updatedData
          });
        }
      } catch (error) {
        if (error.statusCode !== 404) {
          throw error;
        }
        // Ignore not found errors for soft delete
      }
    }
  }
  async getWorkflows(params, spaceId) {
    var _searchResponse$hits$;
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    const {
      limit = 20,
      page = 1,
      enabled,
      createdBy,
      query
    } = params;
    const from = (page - 1) * limit;
    const must = [];

    // Filter by spaceId
    must.push({
      term: {
        spaceId
      }
    });

    // Exclude soft-deleted workflows
    must.push({
      bool: {
        must_not: {
          exists: {
            field: 'deleted_at'
          }
        }
      }
    });
    if (enabled !== undefined && enabled.length > 0) {
      must.push({
        terms: {
          enabled
        }
      });
    }
    if (createdBy && createdBy.length > 0) {
      must.push({
        terms: {
          createdBy
        }
      });
    }
    if (query) {
      must.push({
        bool: {
          should: [
          // Exact phrase matching with boost (text fields only)
          {
            multi_match: {
              query,
              fields: ['name^3', 'description^2'],
              type: 'phrase',
              boost: 3
            }
          },
          // Word-level matching (all fields)
          {
            multi_match: {
              query,
              fields: ['name^2', 'description', 'tags'],
              type: 'best_fields',
              boost: 2
            }
          },
          // Prefix matching for partial word matches (text fields only)
          {
            multi_match: {
              query,
              fields: ['name^2', 'description'],
              type: 'phrase_prefix',
              boost: 1.5
            }
          },
          // Wildcard matching for more flexible partial matches
          {
            bool: {
              should: [{
                wildcard: {
                  'name.keyword': {
                    value: `*${query}*`,
                    case_insensitive: true,
                    boost: 1
                  }
                }
              }, {
                wildcard: {
                  'description.keyword': {
                    value: `*${query}*`,
                    case_insensitive: true,
                    boost: 0.5
                  }
                }
              }, {
                wildcard: {
                  tags: {
                    value: `*${query}*`,
                    case_insensitive: true,
                    boost: 0.5
                  }
                }
              }]
            }
          }],
          minimum_should_match: 1
        }
      });
    }
    const searchResponse = await this.workflowStorage.getClient().search({
      size: limit,
      from,
      track_total_hits: true,
      query: {
        bool: {
          must
        }
      },
      sort: [{
        updated_at: {
          order: 'desc'
        }
      }]
    });
    const workflows = searchResponse.hits.hits.map(hit => {
      if (!hit._source) {
        throw new Error('Missing _source in search result');
      }
      const workflow = this.transformStorageDocumentToWorkflowDto(hit._id, hit._source);
      return {
        ...workflow,
        description: workflow.description || '',
        definition: workflow.definition,
        history: [] // Will be populated below
      };
    }).filter(workflow => workflow !== null);

    // Fetch recent execution history for all workflows
    if (workflows.length > 0) {
      const workflowIds = workflows.map(w => w.id);
      const executionHistory = await this.getRecentExecutionsForWorkflows(workflowIds, spaceId);

      // Populate history for each workflow
      workflows.forEach(workflow => {
        workflow.history = executionHistory[workflow.id] || [];
      });
    }
    return {
      _pagination: {
        page,
        limit,
        total: typeof searchResponse.hits.total === 'number' ? searchResponse.hits.total : ((_searchResponse$hits$ = searchResponse.hits.total) === null || _searchResponse$hits$ === void 0 ? void 0 : _searchResponse$hits$.value) || 0
      },
      results: workflows
    };
  }
  async getWorkflowStats(spaceId) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    const statsResponse = await this.workflowStorage.getClient().search({
      size: 0,
      track_total_hits: true,
      query: {
        bool: {
          must: [{
            term: {
              spaceId
            }
          }],
          must_not: {
            exists: {
              field: 'deleted_at'
            }
          }
        }
      },
      aggs: {
        enabled_count: {
          filter: {
            term: {
              enabled: true
            }
          }
        },
        disabled_count: {
          filter: {
            term: {
              enabled: false
            }
          }
        }
      }
    });
    const aggs = statsResponse.aggregations;

    // Get execution history stats for the last 30 days
    const executionStats = await this.getExecutionHistoryStats(spaceId);
    return {
      workflows: {
        enabled: aggs.enabled_count.doc_count,
        disabled: aggs.disabled_count.doc_count
      },
      executions: executionStats
    };
  }
  async getExecutionHistoryStats(spaceId) {
    try {
      var _response$aggregation, _response$aggregation2;
      const thirtyDaysAgo = new Date();
      thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
      const response = await this.esClient.search({
        index: this.workflowsExecutionIndex,
        size: 0,
        query: {
          bool: {
            must: [{
              range: {
                createdAt: {
                  gte: thirtyDaysAgo.toISOString()
                }
              }
            }, {
              term: {
                spaceId
              }
            }]
          }
        },
        aggs: {
          daily_stats: {
            date_histogram: {
              field: 'createdAt',
              calendar_interval: 'day',
              format: 'yyyy-MM-dd'
            },
            aggs: {
              completed: {
                filter: {
                  term: {
                    status: 'completed'
                  }
                }
              },
              failed: {
                filter: {
                  term: {
                    status: 'failed'
                  }
                }
              },
              cancelled: {
                filter: {
                  term: {
                    status: 'cancelled'
                  }
                }
              }
            }
          }
        }
      });
      const buckets = ((_response$aggregation = response.aggregations) === null || _response$aggregation === void 0 ? void 0 : (_response$aggregation2 = _response$aggregation.daily_stats) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) || [];
      return buckets.map(bucket => ({
        date: bucket.key_as_string,
        timestamp: bucket.key,
        completed: bucket.completed.doc_count,
        failed: bucket.failed.doc_count,
        cancelled: bucket.cancelled.doc_count
      }));
    } catch (error) {
      this.logger.error('Failed to get execution history stats', error);
      return [];
    }
  }
  async getWorkflowAggs(fields, spaceId) {
    if (!this.workflowStorage) {
      throw new Error('WorkflowsService not initialized');
    }
    const aggs = {};
    fields.forEach(field => {
      aggs[field] = {
        terms: {
          field: field === 'name' ? 'name.keyword' : field,
          size: 100
        }
      };
    });
    const aggsResponse = await this.workflowStorage.getClient().search({
      size: 0,
      track_total_hits: true,
      query: {
        bool: {
          must: [{
            term: {
              spaceId
            }
          }],
          must_not: {
            exists: {
              field: 'deleted_at'
            }
          }
        }
      },
      aggs
    });
    const result = {};
    const responseAggs = aggsResponse.aggregations;
    fields.forEach(field => {
      if (responseAggs[field]) {
        result[field] = responseAggs[field].buckets.map(bucket => ({
          label: bucket.key_as_string,
          key: bucket.key,
          doc_count: bucket.doc_count
        }));
      }
    });
    return result;
  }

  // Helper methods remain the same as they don't interact with SavedObjects
  async getWorkflowExecution(executionId, spaceId) {
    return (0, _get_workflow_execution.getWorkflowExecution)({
      esClient: this.esClient,
      logger: this.logger,
      workflowExecutionIndex: this.workflowsExecutionIndex,
      stepsExecutionIndex: this.stepsExecutionIndex,
      workflowExecutionId: executionId,
      spaceId
    });
  }
  async getWorkflowExecutions(params, spaceId) {
    const must = [{
      term: {
        workflowId: params.workflowId
      }
    }, {
      bool: {
        should: [{
          term: {
            spaceId
          }
        },
        // Backward compatibility for objects without spaceId
        {
          bool: {
            must_not: {
              exists: {
                field: 'spaceId'
              }
            }
          }
        }],
        minimum_should_match: 1
      }
    }];
    if (params.statuses) {
      must.push({
        terms: {
          status: params.statuses
        }
      });
    }
    if (params.executionTypes) {
      must.push({
        terms: {
          executionType: params.executionTypes
        }
      });
    }
    return (0, _search_workflow_executions.searchWorkflowExecutions)({
      esClient: this.esClient,
      logger: this.logger,
      workflowExecutionIndex: this.workflowsExecutionIndex,
      query: {
        bool: {
          must
        }
      }
    });
  }
  async getWorkflowExecutionHistory(executionId, spaceId) {
    const response = await this.esClient.search({
      index: this.stepsExecutionIndex,
      query: {
        bool: {
          must: [{
            term: {
              executionId
            }
          }, {
            term: {
              spaceId
            }
          }]
        }
      },
      sort: [{
        timestamp: {
          order: 'asc'
        }
      }]
    });
    return response.hits.hits.map(hit => {
      const source = hit._source;
      const startedAt = source.startedAt;
      const finishedAt = source.endedAt || source.finishedAt;

      // Calculate duration in milliseconds if both timestamps are available
      let duration = 0;
      if (startedAt && finishedAt) {
        duration = new Date(finishedAt).getTime() - new Date(startedAt).getTime();
      }
      return {
        ...source,
        finishedAt: finishedAt || '',
        duration
      };
    });
  }

  /**
   * Efficiently fetch the most recent execution for multiple workflows
   */
  async getRecentExecutionsForWorkflows(workflowIds, spaceId) {
    if (!this.esClient || workflowIds.length === 0) {
      return {};
    }
    try {
      var _response$aggregation3;
      const response = await this.esClient.search({
        index: this.workflowsExecutionIndex,
        size: 0,
        // We only want aggregations
        query: {
          bool: {
            must: [{
              terms: {
                workflowId: workflowIds
              }
            }, {
              bool: {
                should: [{
                  term: {
                    spaceId
                  }
                },
                // Backward compatibility for objects without spaceId
                {
                  bool: {
                    must_not: {
                      exists: {
                        field: 'spaceId'
                      }
                    }
                  }
                }],
                minimum_should_match: 1
              }
            }]
          }
        },
        aggs: {
          workflows: {
            terms: {
              field: 'workflowId',
              size: workflowIds.length
            },
            aggs: {
              recent_executions: {
                top_hits: {
                  size: 1,
                  // Get only the most recent execution per workflow
                  sort: [{
                    finishedAt: {
                      order: 'desc'
                    }
                  }]
                }
              }
            }
          }
        }
      });
      const result = {};
      if ((_response$aggregation3 = response.aggregations) !== null && _response$aggregation3 !== void 0 && _response$aggregation3.workflows && 'buckets' in response.aggregations.workflows) {
        const buckets = response.aggregations.workflows.buckets;
        buckets.forEach(bucket => {
          const workflowId = bucket.key;
          const hits = bucket.recent_executions.hits.hits;
          if (hits.length > 0) {
            var _execution$workflowDe;
            const execution = hits[0]._source;
            result[workflowId] = [{
              id: execution.id,
              workflowId: execution.workflowId,
              workflowName: ((_execution$workflowDe = execution.workflowDefinition) === null || _execution$workflowDe === void 0 ? void 0 : _execution$workflowDe.name) || 'Unknown Workflow',
              status: execution.status,
              startedAt: execution.startedAt,
              finishedAt: execution.finishedAt || execution.startedAt,
              duration: execution.finishedAt && execution.startedAt ? new Date(execution.finishedAt).getTime() - new Date(execution.startedAt).getTime() : null
            }];
          }
        });
      }
      return result;
    } catch (error) {
      this.logger.error(`Failed to fetch recent executions for workflows: ${error}`);
      return {};
    }
  }
  async getStepExecutions(params, spaceId) {
    return (0, _search_step_executions.searchStepExecutions)({
      esClient: this.esClient,
      logger: this.logger,
      stepsExecutionIndex: this.stepsExecutionIndex,
      workflowExecutionId: params.executionId,
      additionalQuery: {
        term: {
          id: params.id
        }
      },
      spaceId
    });
  }
  async getExecutionLogs(params, spaceId) {
    return this.workflowEventLoggerService.searchLogs(params, spaceId);
  }
  async getStepLogs(params, spaceId) {
    return this.workflowEventLoggerService.searchLogs(params, spaceId);
  }
  getLogger() {
    return this.workflowEventLoggerService;
  }
  async getStepExecution(params, spaceId) {
    const {
      executionId,
      id
    } = params;
    const response = await this.esClient.search({
      index: this.stepsExecutionIndex,
      query: {
        bool: {
          must: [{
            term: {
              workflowRunId: executionId
            }
          }, {
            term: {
              id
            }
          }, {
            term: {
              spaceId
            }
          }]
        }
      },
      size: 1,
      track_total_hits: false
    });
    if (response.hits.hits.length === 0) {
      return null;
    }
    return response.hits.hits[0]._source;
  }
  transformStorageDocumentToWorkflowDto(id, source) {
    return {
      id,
      name: source.name,
      description: source.description,
      enabled: source.enabled,
      yaml: source.yaml,
      definition: source.definition,
      createdBy: source.createdBy,
      lastUpdatedBy: source.lastUpdatedBy,
      valid: source.valid,
      createdAt: new Date(source.created_at),
      lastUpdatedAt: new Date(source.updated_at)
    };
  }
  generateWorkflowId() {
    return `workflow-${(0, _uuid.v4)()}`;
  }
}
exports.WorkflowsService = WorkflowsService;